package liveline

import (
	"context"
	"os"
	"sort"
	"sync"

	"code.justin.tv/discovery/liveline/proto/liveline"
	"code.justin.tv/foundation/twitchclient"
	"github.com/sirupsen/logrus"
)

const (
	livelineQueryBatchSize = 2000
)

// livelineClient wraps the liveline client.
type livelineClient struct {
	livelineClient liveline.Liveline
}

// Liveline interface for Mockery.
type Liveline interface {
	GetStreamsByChannelIDs(ctx context.Context, ids []string) []string
}

type liveChannel struct {
	ID          string
	ViewerCount int64
}

// NewClient instantiates a new Liveline Client.
func NewClient() Liveline {
	livelineConf := twitchclient.ClientConf{
		Host: os.Getenv("LIVELINE_HOST"),
	}
	return &livelineClient{liveline.NewLivelineProtobufClient(livelineConf.Host, twitchclient.NewHTTPClient(livelineConf))}
}

func (l *livelineClient) GetStreamsByChannelIDs(ctx context.Context, ids []string) []string {
	var idToViewerCount sync.Map
	var wg sync.WaitGroup

	for low := 0; low < len(ids); low += livelineQueryBatchSize {
		// query liveline for live channels in batches
		high := low + livelineQueryBatchSize
		if high > len(ids) {
			high = len(ids)
		}

		wg.Add(1)
		go func(low, high int) {
			defer wg.Done()

			localIDs := ids[low:high]

			liveStreams, err := l.livelineClient.GetStreamsByChannelIDs(ctx, &liveline.StreamsByChannelIDsRequest{
				ChannelIds: localIDs,
				SortKey:    liveline.SortField_CCV,
				Order:      liveline.SortOrder_DESC,
			})
			if err != nil {
				// do not fail the request if one of the liveline calls fails
				logrus.WithError(err).Error("liveline: failed to query live streams")
				return
			}

			// populate idToViewerCount with IDs of all live channels and their viewer counts
			for _, stream := range liveStreams.Streams {
				idToViewerCount.Store(stream.ChannelId, stream.ViewcountData.Viewcount)
			}
		}(low, high)
	}

	wg.Wait()

	liveChannels := make([]liveChannel, 0, len(ids))
	for _, id := range ids {
		if countValue, ok := idToViewerCount.Load(id); ok {
			liveChannels = append(liveChannels, liveChannel{
				ID:          id,
				ViewerCount: countValue.(int64),
			})
		}
	}

	// sort by descending view count
	sort.SliceStable(liveChannels, func(i, j int) bool { return liveChannels[i].ViewerCount > liveChannels[j].ViewerCount })

	liveIDs := make([]string, len(liveChannels))
	for idx, liveChannel := range liveChannels {
		liveIDs[idx] = liveChannel.ID
	}
	return liveIDs
}
