package postgres

import (
	"database/sql"
	"fmt"
	"strings"

	"github.com/pkg/errors"
)

// DB is a wrapper for sql.DB.
type DB struct {
	*sql.DB
}

// DBConfig are the configuration values needed to connect to PostgreSQL.
type DBConfig struct {
	Host     string
	Port     string
	DBName   string
	User     string
	Password string
	SSLMode  string
}

// NewDB opens a connection pool to the PostgreSQL database,
// specified by the database credentials and address.
//
// Pinging the database ensures that the connection is established.
func NewDB(config DBConfig) (*DB, error) {
	dbConn, err := sql.Open("postgres", connectionString(config))
	if err != nil {
		return nil, errors.Wrap(err, "postgres: failed to open sql connection")
	}

	if err := dbConn.Ping(); err != nil {
		return nil, errors.Wrap(err, "postgres: failed to ping database")
	}

	return &DB{dbConn}, nil
}

func connectionString(config DBConfig) string {
	params := []string{}

	if len(config.Host) != 0 {
		params = append(params, "host="+config.Host)
	}

	if len(config.Port) != 0 {
		params = append(params, "port="+config.Port)
	}

	if len(config.DBName) != 0 {
		params = append(params, "dbname="+config.DBName)
	}

	if len(config.User) != 0 {
		params = append(params, "user="+config.User)
	}

	if len(config.Password) != 0 {
		params = append(params, fmt.Sprintf("password='%s'", config.Password))
	}

	if len(config.SSLMode) != 0 {
		params = append(params, "sslmode="+config.SSLMode)
	}

	return strings.Join(params, " ")
}
