package s3

import (
	"context"
	"strings"

	"code.justin.tv/cb/roster/internal/image"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/pkg/errors"
)

// Image contains the metadata of an image object stored in S3.
type Image struct {
	TeamName string
	Category string
	ID       string
	Format   string
}

// FindParams contains the required fields for Client.Find.
type FindParams struct {
	TeamName string
	Category string
	ID       string
}

// Find attempts find an image object stored in S3 by the image's prefix.
func (c *Client) Find(ctx context.Context, params FindParams) (Image, error) {
	prefix := sourcePrefix(params)

	output, err := c.service.ListObjectsV2WithContext(ctx, &s3.ListObjectsV2Input{
		Bucket: aws.String(c.sourceBucket),
		Prefix: aws.String(prefix),
	})

	if err != nil {
		return Image{}, errors.Wrap(err, "s3: failed to list images from source")
	}

	if len(output.Contents) == 0 {
		return Image{}, ErrNoImage
	}

	if output.Contents[0].Key == nil {
		return Image{}, ErrMalformedImage
	}

	return Image{
		Category: params.Category,
		Format:   strings.Replace(*output.Contents[0].Key, prefix, "", 1),
		ID:       params.ID,
		TeamName: params.TeamName,
	}, nil
}

func sourcePrefix(params FindParams) string {
	return image.Key(params.Category, params.TeamName, params.ID, "")
}
