package s3

import (
	"context"
	"errors"

	"code.justin.tv/cb/roster/internal/s3/mocks"
	"github.com/aws/aws-sdk-go/service/s3"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("Find", func() {
	var (
		client   Client
		mockedS3 *mocks.Service
	)

	const (
		sourceBucket = "source"
		category     = "background"
		id           = "uuid1234"
		teamName     = "some-team"
		format       = "jpeg"
	)

	BeforeEach(func() {
		mockedS3 = &mocks.Service{}

		client = Client{
			service:      mockedS3,
			sourceBucket: sourceBucket,
		}
	})

	Context("when finding an existing image fails", func() {
		BeforeEach(func() {
			mockedS3.On("ListObjectsV2WithContext", mock.Anything, mock.Anything).
				Return(nil, errors.New("some error"))
		})

		It("returns an error", func() {
			_, err := client.Find(context.Background(), FindParams{
				Category: category,
				ID:       id,
				TeamName: teamName,
			})

			mockedS3.AssertExpectations(GinkgoT())

			Expect(err).To(HaveOccurred())
			Expect(err.Error()).To(ContainSubstring("some error"))
			Expect(err.Error()).To(ContainSubstring("failed to list images from source"))
		})
	})

	Context("when finding an existing image returns no results", func() {
		BeforeEach(func() {
			mockedS3.On("ListObjectsV2WithContext", mock.Anything, mock.Anything).
				Return(&s3.ListObjectsV2Output{
					Contents: nil,
				}, nil)
		})

		It("returns an error", func() {
			_, err := client.Find(context.Background(), FindParams{
				Category: category,
				ID:       id,
				TeamName: teamName,
			})

			mockedS3.AssertExpectations(GinkgoT())

			Expect(err).To(HaveOccurred())
			Expect(err).To(Equal(ErrNoImage))
		})
	})

	Context("when finding an existing image returns an invalid image key", func() {
		BeforeEach(func() {
			mockedS3.On("ListObjectsV2WithContext", mock.Anything, mock.Anything).
				Return(&s3.ListObjectsV2Output{
					Contents: []*s3.Object{
						{Key: nil},
					},
				}, nil)
		})

		It("returns an error", func() {
			_, err := client.Find(context.Background(), FindParams{
				Category: category,
				ID:       id,
				TeamName: teamName,
			})

			mockedS3.AssertExpectations(GinkgoT())

			Expect(err).To(HaveOccurred())
			Expect(err).To(Equal(ErrMalformedImage))
		})
	})

	Context("when finding an existing image succeeds", func() {
		BeforeEach(func() {
			key := "images/background/some-team/uuid1234.jpeg"

			mockedS3.On("ListObjectsV2WithContext", mock.Anything, mock.Anything).
				Return(&s3.ListObjectsV2Output{
					Contents: []*s3.Object{
						{Key: &key},
					},
				}, nil)
		})

		It("returns no error", func() {
			image, err := client.Find(context.Background(), FindParams{
				Category: category,
				ID:       id,
				TeamName: teamName,
			})

			mockedS3.AssertExpectations(GinkgoT())

			Expect(err).NotTo(HaveOccurred())
			Expect(image.TeamName).To(Equal(teamName))
			Expect(image.Category).To(Equal(category))
			Expect(image.ID).To(Equal(id))
			Expect(image.Format).To(Equal(format))
		})
	})
})
