package s3

import (
	"context"
	"fmt"

	"code.justin.tv/cb/roster/internal/image"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/pkg/errors"
)

// SaveParams contains the required fields for Client.Save.
type SaveParams struct {
	TeamName string
	Category string
	ID       string
	Format   string
}

// Save copies an existing image object from the source bucket to the destination bucket.
func (c *Client) Save(ctx context.Context, params SaveParams) error {
	_, err := c.service.CopyObjectWithContext(ctx, &s3.CopyObjectInput{
		ACL:         aws.String("public-read"),
		Bucket:      aws.String(c.destinationBucket),
		ContentType: aws.String("image/" + params.Format),
		CopySource:  aws.String(c.copySource(params)),
		Key:         aws.String(image.LegacyKey(params.Category, params.TeamName, params.ID, params.Format)),
	})

	if err != nil {
		if awsError, ok := err.(awserr.Error); ok {
			if awsError.Code() == s3.ErrCodeNoSuchKey {
				return ErrNoImage
			}
		}

		return errors.Wrap(err, "s3: failed to copy image")
	}

	return nil
}

func (c *Client) copySource(params SaveParams) string {
	key := image.Key(params.Category, params.TeamName, params.ID, params.Format)

	return fmt.Sprintf("/%s/%s", c.sourceBucket, key)
}
