package s3

import (
	"context"
	"errors"

	"code.justin.tv/cb/roster/internal/s3/mocks"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/service/s3"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("Save", func() {
	var (
		client     Client
		mockedS3   *mocks.Service
		saveParams SaveParams
	)

	const (
		sourceBucket      = "source"
		destinationBucket = "destination"
		category          = "background"
		id                = "uuid1234"
		teamName          = "some-team"
		format            = "jpeg"
	)

	BeforeEach(func() {
		mockedS3 = &mocks.Service{}

		client = Client{
			service:           mockedS3,
			sourceBucket:      sourceBucket,
			destinationBucket: destinationBucket,
		}

		saveParams = SaveParams{
			TeamName: teamName,
			Category: category,
			ID:       id,
			Format:   format,
		}
	})

	Context("when the image for copy is missing", func() {
		BeforeEach(func() {
			copyError := awserr.New(s3.ErrCodeNoSuchKey, "some message", errors.New("some error"))

			mockedS3.On("CopyObjectWithContext", mock.Anything, mock.Anything).
				Return(nil, copyError)
		})

		It("return an error", func() {
			err := client.Save(context.Background(), saveParams)
			Expect(err).To(HaveOccurred())
			Expect(err).To(Equal(ErrNoImage))

			mockedS3.AssertExpectations(GinkgoT())
		})
	})

	Context("when copying the image fails", func() {
		BeforeEach(func() {
			mockedS3.On("CopyObjectWithContext", mock.Anything, mock.Anything).
				Return(nil, errors.New("some copy error"))
		})

		It("return an error", func() {
			err := client.Save(context.Background(), saveParams)
			Expect(err).To(HaveOccurred())
			Expect(err.Error()).To(ContainSubstring("some copy error"))

			mockedS3.AssertExpectations(GinkgoT())
		})
	})

	Context("when copying the image succeeds", func() {
		BeforeEach(func() {
			copyInput := &s3.CopyObjectInput{
				ACL:         aws.String("public-read"),
				Bucket:      aws.String("destination"),
				ContentType: aws.String("image/jpeg"),
				CopySource:  aws.String("/source/images/background/some-team/uuid1234.jpeg"),
				Key:         aws.String("team-some-team-background_image-uuid1234.jpeg"),
			}

			mockedS3.On("CopyObjectWithContext", mock.Anything, copyInput).
				Return(&s3.CopyObjectOutput{}, nil)
		})

		It("return no error", func() {
			err := client.Save(context.Background(), saveParams)
			Expect(err).NotTo(HaveOccurred())

			mockedS3.AssertExpectations(GinkgoT())
		})
	})
})
