package uploader

import (
	"strings"
	"time"

	"code.justin.tv/cb/roster/internal/image"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/pkg/errors"
)

// The following errors are returned when validating ImageUploadParams.
var (
	ErrContentLengthOverLimit = errors.New("uploader: content length over limit")
	ErrInvalidCategory        = errors.New("uploader: invalid image category")
	ErrMissingContentLength   = errors.New("uploader: content length cannot be empty")
	ErrInvalidContentType     = errors.New("uploader: content type cannot be empty")
	ErrMissingTeamName        = errors.New("uploader: team name cannot be empty")
)

const uploadExpiration = 1 * time.Minute

// ImageUpload contains the ID of the image to-be-uploaded,
// as well as the URL for uploading the image file.
type ImageUpload struct {
	ImageID string
	URL     string
}

// NewImageUpload creates a pre-signed URL for uploading directly to an S3 bucket.
// Uploaded objects are segregated by team ID.
func (c *Client) NewImageUpload(params ImageUploadParams) (ImageUpload, error) {
	if err := params.validate(); err != nil {
		return ImageUpload{}, err
	}

	return c.presignImageUpload(params)
}

func (c *Client) presignImageUpload(params ImageUploadParams) (ImageUpload, error) {
	imageID := image.NewID()
	key := image.Key(params.Category, params.TeamName, imageID, parseFormatFromContentType(params.ContentType))

	req, _ := c.service.PutObjectRequest(&s3.PutObjectInput{
		Bucket:        aws.String(c.bucket),
		ContentLength: aws.Int64(int64(params.ContentLength)),
		ContentType:   aws.String(params.ContentType),
		Key:           aws.String(key),
	})

	url, err := req.Presign(uploadExpiration)
	if err != nil {
		return ImageUpload{}, errors.Wrap(err, "s3: failed to presign upload request")
	}

	return ImageUpload{
		ImageID: imageID,
		URL:     url,
	}, nil
}

// ImageUploadParams contains the required parameters for Uploader.NewImageUpload.
type ImageUploadParams struct {
	Category      string
	ContentLength uint64
	ContentType   string
	TeamName      string
}

func (i *ImageUploadParams) validate() error {
	if ok := image.ValidCategory(i.Category); !ok {
		return ErrInvalidCategory
	}

	if i.ContentLength == 0 {
		return ErrMissingContentLength
	}

	if i.ContentLength > image.MaxSize {
		return ErrContentLengthOverLimit
	}

	format := parseFormatFromContentType(i.ContentType)
	if ok := image.ValidFormat(format); !ok {
		return ErrInvalidContentType
	}

	if len(i.TeamName) == 0 {
		return ErrMissingTeamName
	}

	return nil
}

func parseFormatFromContentType(contentType string) string {
	return strings.TrimPrefix(contentType, "image/")
}
