package uploader

import (
	"net/http"

	"code.justin.tv/cb/roster/internal/s3/uploader/mocks"
	"github.com/aws/aws-sdk-go/aws/request"
	"github.com/aws/aws-sdk-go/service/s3"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/satori/go.uuid"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("NewImageUpload", func() {
	var (
		client   Client
		mockedS3 *mocks.Service
	)

	const (
		category      = "background"
		contentLength = 500000
		contentType   = "image/jpeg"
		teamName      = "some team"
	)

	BeforeEach(func() {
		mockedS3 = &mocks.Service{}

		client = Client{
			service: mockedS3,
		}
	})

	Context("with invalid parameters", func() {
		It("errors when category is invalid", func() {
			_, err := client.NewImageUpload(ImageUploadParams{
				Category:      "not a valid category",
				ContentLength: contentLength,
				ContentType:   contentType,
				TeamName:      teamName,
			})

			Expect(err).To(HaveOccurred())
			Expect(err).To(Equal(ErrInvalidCategory))
		})

		It("errors when content length is zero", func() {
			_, err := client.NewImageUpload(ImageUploadParams{
				Category:      category,
				ContentLength: 0,
				ContentType:   contentType,
				TeamName:      teamName,
			})

			Expect(err).To(HaveOccurred())
			Expect(err).To(Equal(ErrMissingContentLength))
		})

		It("errors when content length is invalid", func() {
			_, err := client.NewImageUpload(ImageUploadParams{
				Category:      category,
				ContentLength: contentLength + 1,
				ContentType:   contentType,
				TeamName:      teamName,
			})

			Expect(err).To(HaveOccurred())
			Expect(err).To(Equal(ErrContentLengthOverLimit))
		})

		It("errors when content type is invalid", func() {
			_, err := client.NewImageUpload(ImageUploadParams{
				Category:      category,
				ContentLength: contentLength,
				ContentType:   "not jpeg or png",
				TeamName:      teamName,
			})

			Expect(err).To(HaveOccurred())
			Expect(err).To(Equal(ErrInvalidContentType))
		})

		It("errors when team name is missing", func() {
			_, err := client.NewImageUpload(ImageUploadParams{
				Category:      category,
				ContentLength: contentLength,
				ContentType:   contentType,
				TeamName:      "",
			})

			Expect(err).To(HaveOccurred())
			Expect(err).To(Equal(ErrMissingTeamName))
		})
	})

	Context("with valid parameters", func() {
		It("returns valid image id and upload url", func() {
			httpReq, err := http.NewRequest(http.MethodPut, "http://h.ost/path/etc.xyz?hello=hi", nil)
			Expect(err).NotTo(HaveOccurred())

			mockedS3.On("PutObjectRequest", mock.Anything).Return(&request.Request{
				Operation:   &request.Operation{},
				HTTPRequest: httpReq,
			}, &s3.PutObjectOutput{})

			upload, err := client.NewImageUpload(ImageUploadParams{
				Category:      category,
				ContentLength: contentLength,
				ContentType:   contentType,
				TeamName:      teamName,
			})

			mockedS3.AssertExpectations(GinkgoT())

			Expect(err).NotTo(HaveOccurred())

			_, err = uuid.FromString(upload.ImageID)
			Expect(err).NotTo(HaveOccurred())

			Expect(upload.URL).To(Equal("http://h.ost/path/etc.xyz?hello=hi"))
		})
	})
})
