package worker

import (
	"context"
	"encoding/json"
	"net/http"
	"time"

	"code.justin.tv/cb/roster/internal/db"
	log "github.com/sirupsen/logrus"
)

// The SQS message contains a number of other metadata fields, but we don't
// use them so we only decode the message body here.
type requestBody struct {
	Message string
}

type hardDeleteUserMessage struct {
	UserID    string    `json:"user_id"`
	Timestamp time.Time `json:"timestamp"`
}

func (s *Server) process(w http.ResponseWriter, req *http.Request) {
	var reqBody requestBody
	err := json.NewDecoder(req.Body).Decode(&reqBody)
	if err != nil {
		log.WithError(err).Warn("worker: failed to parse request body")
		w.WriteHeader(http.StatusBadRequest)
		return
	}

	var messageBody hardDeleteUserMessage
	err = json.Unmarshal([]byte(reqBody.Message), &messageBody)
	if err != nil {
		log.WithError(err).WithField("message", reqBody.Message).Warn("worker: failed to parse message body")
		w.WriteHeader(http.StatusBadRequest)
		return
	}
	userID := messageBody.UserID

	// 1. Get all memberships
	memberships, err := s.dbWriter.GetChannelMemberships(req.Context(), userID)
	if err != nil {
		log.WithError(err).WithField("user_id", userID).Error("worker: failed to get user memberships")
		w.WriteHeader(http.StatusInternalServerError)
		return
	}

	// 2. Delete all user memberships
	err = s.dbWriter.DeleteMemberships(req.Context(), userID)
	if err != nil && err != db.ErrNoRowFoundForDeletion {
		log.WithError(err).WithField("user_id", userID).Error("worker: failed to delete memberships for user")
		w.WriteHeader(http.StatusInternalServerError)
		return
	}

	// 3. Clear cached memberships
	for _, membership := range memberships {
		go func(teamID string) {
			cacheError := s.cache.ClearAllTeamMembershipsForTeam(context.Background(), teamID)
			if cacheError != nil {
				log.WithError(cacheError).WithField("user_id", userID).Error("worker: failed to clear cached team memberships")
			}
		}(membership.TeamID)
	}

	go func() {
		cacheError := s.cache.ClearChannelMemberships(context.Background(), userID)
		if cacheError != nil {
			log.WithError(cacheError).WithField("user_id", userID).Error("worker: failed to clear cached channel memberships")
		}
	}()

	// 4. Delete featured
	err = s.dbWriter.DeleteAllFeaturedByChannel(req.Context(), userID)
	if err != nil && err != db.ErrNoRowFoundForDeletion {
		log.WithError(err).WithField("user_id", userID).Error("worker: failed to delete featured for user")
		w.WriteHeader(http.StatusInternalServerError)
		return
	}

	// 5. Delete invitations
	err = s.dbWriter.DeleteChannelInvitations(req.Context(), userID)
	if err != nil && err != db.ErrNoRowFoundForDeletion {
		log.WithError(err).WithField("user_id", userID).Error("worker: failed to delete invitations for user")
		w.WriteHeader(http.StatusInternalServerError)
		return
	}

	w.WriteHeader(http.StatusOK)
}
