resource "aws_iam_role" "app" {
  name = "${var.team_code}-${var.name}-${var.environment}"

  assume_role_policy = <<POLICY
{
  "Version": "2012-10-17",
  "Statement": {
    "Action": "sts:AssumeRole",
    "Effect": "Allow",
    "Principal": {
      "Service": "ec2.amazonaws.com"
    }
  }
}
POLICY

}

resource "aws_iam_role_policy" "pdms" {
  name = "pdms-policy"
  role = aws_iam_role.app.name

  policy = <<POLICY
{
    "Version": "2012-10-17",
    "Statement": [
        {
            "Sid": "",
            "Effect": "Allow",
            "Action": "sts:AssumeRole",
            "Resource": [
                "${var.pdms_role_arn}"
            ]
        }
    ]
}

POLICY

}

# The Sandstorm policy allows the Sandstorm Agent to assume the role
# of the application.
resource "aws_iam_role_policy" "sandstorm" {
  name = "sandstorm-policy"
  role = aws_iam_role.app.name

  policy = <<POLICY
{
  "Version": "2012-10-17",
  "Statement": {
    "Action": "sts:AssumeRole",
    "Effect": "Allow",
    "Resource": "${var.sandstorm_iam_role_arn}"
  }
}
POLICY

}

# The CloudWatch logs policy allows:
# - Create groups in CloudWatch logs for proxy and deployment logs
# - Stream logs from each EC2 instance
resource "aws_iam_role_policy" "cloudwatch-logs" {
  name = "cloudwatch-logs-policy"
  role = aws_iam_role.app.name

  policy = <<POLICY
{
  "Version": "2012-10-17",
  "Statement": {
    "Effect": "Allow",
    "Action": [
      "logs:CreateLogStream",
      "logs:PutLogEvents"
    ],
    "Resource": [
      "arn:aws:logs:us-west-2:*:*",
      "arn:aws:logs:*:*:log-group:/aws/elasticbeanstalk*"
    ]
  }
}
POLICY

}

# The CloudWatch put metric data policy allows:
# - Put metric data to CloudWatch (cloudwatch metrics have no resources to filter on)
resource "aws_iam_role_policy" "cloudwatch-metric-access" {
  name = "cloudwatch-metric-access-policy"
  role = aws_iam_role.app.name

  policy = <<POLICY
{
  "Version": "2012-10-17",
  "Statement": {
    "Effect": "Allow",
    "Action": [
      "cloudwatch:PutMetricData"
    ],
    "Resource": [
      "*"
    ]
  }
}
POLICY

}

resource "aws_iam_role_policy" "s3-uploads" {
  name = "s3-uploads-policy"
  role = aws_iam_role.app.name

  policy = <<POLICY
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
        "s3:GetBucketLocation",
        "s3:ListBucket"
      ],
      "Resource": "${var.s3_uploads_iam_role_arn}"
    },
    {
      "Effect": "Allow",
      "Action": [
        "s3:GetObject",
        "s3:GetObjectAcl",
        "s3:PutObject"
      ],
      "Resource": "${var.s3_uploads_iam_role_arn}/*"
    }
  ]
}
POLICY

}

resource "aws_iam_role_policy" "sqs" {
  name = "sqs-assets-policy"
  role = aws_iam_role.app.name

  policy = <<POLICY
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Action": [
        "sqs:ChangeMessageVisibility",
        "sqs:DeleteMessage",
        "sqs:ReceiveMessage",
        "sqs:SendMessage"
      ],
      "Effect": "Allow",
      "Resource": "${aws_sqs_queue.worker_queue.arn}"
    }
  ]
}
POLICY

}

resource "aws_iam_role_policy" "s3-assets" {
  name = "s3-assets-policy"
  role = aws_iam_role.app.name

  policy = <<POLICY
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
        "s3:GetObject",
        "s3:PutObject",
        "s3:DeleteObject"
      ],
      "Resource": "${var.s3_assets_iam_role_arn}/*"
    }
  ]
}
POLICY

}

resource "aws_iam_role_policy" "s3-ttv-user-pictures-prod" {
  name = "s3-ttv-user-pictures-prod-write-policy"
  role = aws_iam_role.app.name

  policy = <<POLICY
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
        "s3:PutObject",
        "s3:PutObjectAcl"
      ],
      "Resource": "arn:aws:s3:::ttv-user-pictures-prod/*"
    }
  ]
}
POLICY

}

resource "aws_iam_role_policy" "s2s2-invoke" {
  name = "s2s2-invoke-policy"
  role = aws_iam_role.app.name

  policy = <<POLICY
{
  "Version": "2012-10-17",
  "Statement": [
    {
      "Effect": "Allow",
      "Action": [
        "execute-api:Invoke"
      ],
      "Resource": "arn:aws:execute-api:*:985585625942:*"
    }
  ]
}
POLICY

}

# Instance profile is a container for one single IAM role.
resource "aws_iam_instance_profile" "app" {
  name = aws_iam_role.app.name
  role = aws_iam_role.app.name
}

# Find the EventBusAccess policy for twitch-cb-aws.
data "aws_cloudformation_export" "eventbus_access_policy_arn" {
  name = "eventbus-access-policy-arn"
}

# Attach the EventBusAccess policy to the Roster app role.
resource "aws_iam_role_policy_attachment" "eventbus_access_attach" {
  role       = aws_iam_role.app.name
  policy_arn = data.aws_cloudformation_export.eventbus_access_policy_arn.value
}
