resource "aws_elastic_beanstalk_application" "app" {
  name        = "${var.team_code}-${var.name}-${var.environment}"
  description = "${var.team} team's Go service for Twitch Teams"

  appversion_lifecycle {
    service_role          = "arn:aws:iam::989470033077:role/aws-elasticbeanstalk-service-role"
    max_count             = 20
    delete_source_from_s3 = false
  }
}

output "elastic_beanstalk_app_name" {
  value = aws_elastic_beanstalk_application.app.name
}

resource "aws_elastic_beanstalk_environment" "api" {
  name                = "${aws_elastic_beanstalk_application.app.name}-api"
  application         = aws_elastic_beanstalk_application.app.name
  cname_prefix        = "${aws_elastic_beanstalk_application.app.name}-api"
  solution_stack_name = var.solution_stack_name
  tier                = "WebServer"

  tags = {
    environment = var.environment
    name        = var.name
    owner       = var.owner
    project     = var.project
    team        = var.team
  }

  setting {
    namespace = "aws:elasticbeanstalk:command"
    name      = "DeploymentPolicy"
    value     = "Rolling"
  }

  setting {
    namespace = "aws:elasticbeanstalk:command"
    name      = "BatchSizeType"
    value     = "Percentage"
  }

  setting {
    namespace = "aws:elasticbeanstalk:command"
    name      = "BatchSize"
    value     = var.deployment_batch_size_percentage
  }

  setting {
    namespace = "aws:elasticbeanstalk:application"
    name      = "Application Healthcheck URL"
    value     = "/health"
  }

  setting {
    namespace = "aws:elasticbeanstalk:healthreporting:system"
    name      = "SystemType"
    value     = "enhanced"
  }

  setting {
    namespace = "aws:elasticbeanstalk:healthreporting:system"
    name      = "ConfigDocument"

    value = <<JSON
{
  "CloudWatchMetrics": {
    "Environment": {
      "ApplicationLatencyP50": 60,
      "ApplicationLatencyP90": 60,
      "ApplicationLatencyP99": 60,
      "ApplicationRequests2xx": 60,
      "ApplicationRequests3xx": 60,
      "ApplicationRequests4xx": 60,
      "ApplicationRequests5xx": 60,
      "ApplicationRequestsTotal": 60,
      "InstancesUnknown": 60,
      "InstancesNoData": 60,
      "InstancesSevere": 60,
      "InstancesWarning": 60,
      "InstancesDegraded": 60,
      "InstancesPending": 60,
      "InstancesInfo": 60,
      "InstancesOk": 60
    },
    "Instance": {
      "ApplicationLatencyP50": 60,
      "ApplicationLatencyP90": 60,
      "ApplicationLatencyP99": 60,
      "ApplicationRequests2xx": 60,
      "ApplicationRequests3xx": 60,
      "ApplicationRequests4xx": 60,
      "ApplicationRequests5xx": 60,
      "ApplicationRequestsTotal": 60,
      "CPUIdle": 60,
      "CPUIowait": 60,
      "CPUIrq": 60,
      "CPUNice": 60,
      "CPUSoftirq": 60,
      "CPUSystem": 60,
      "CPUUser": 60,
      "InstanceHealth": 60,
      "LoadAverage1min": 60,
      "LoadAverage5min": 60,
      "RootFilesystemUtil": 60
    }
  },
  "Rules": {
    "Environment": {
      "Application": {
        "ApplicationRequests4xx": {
          "Enabled": false
        }
      }
    }
  },
  "Version": 1
}
JSON

  }

  setting {
    namespace = "aws:elasticbeanstalk:environment"
    name      = "ServiceRole"
    value     = "aws-elasticbeanstalk-service-role"
  }

  # Environment variables
  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "ENVIRONMENT"
    value     = var.environment
  }

  setting {
    namespace = "aws:elasticbeanstalk:application:environment"
    name      = "SANDSTORM_IAM_ROLE_ARN"
    value     = var.sandstorm_iam_role_arn
  }

  // Elastic Load Balance
  setting {
    namespace = "aws:elb:loadbalancer"
    name      = "CrossZone"
    value     = "true"
  }

  setting {
    namespace = "aws:elb:healthcheck"
    name      = "Interval"
    value     = "30"
  }

  setting {
    namespace = "aws:elb:policies"
    name      = "ConnectionDrainingEnabled"
    value     = "true"
  }

  // Autoscaling Group
  setting {
    namespace = "aws:autoscaling:asg"
    name      = "Availability Zones"
    value     = "Any 3"
  }

  setting {
    namespace = "aws:autoscaling:asg"
    name      = "MinSize"
    value     = var.asg_min_size
  }

  setting {
    namespace = "aws:autoscaling:asg"
    name      = "MaxSize"
    value     = var.asg_max_size
  }

  setting {
    namespace = "aws:autoscaling:trigger"
    name      = "MeasureName"
    value     = "CPUUtilization"
  }

  setting {
    namespace = "aws:autoscaling:trigger"
    name      = "Unit"
    value     = "Percent"
  }

  setting {
    namespace = "aws:autoscaling:trigger"
    name      = "LowerThreshold"
    value     = "40"
  }

  setting {
    namespace = "aws:autoscaling:trigger"
    name      = "UpperThreshold"
    value     = "60"
  }

  // Launch configuration
  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "IamInstanceProfile"
    value     = aws_iam_instance_profile.app.id
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "SecurityGroups"
    value     = data.terraform_remote_state.account.outputs.twitch_subnets_sg
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "EC2KeyName"
    value     = "${var.team_code}-${var.name}-${var.environment}"
  }

  setting {
    namespace = "aws:autoscaling:launchconfiguration"
    name      = "InstanceType"
    value     = var.instance_type
  }

  setting {
    namespace = "aws:autoscaling:updatepolicy:rollingupdate"
    name      = "RollingUpdateEnabled"
    value     = "true"
  }

  setting {
    namespace = "aws:autoscaling:updatepolicy:rollingupdate"
    name      = "RollingUpdateType"
    value     = "Health"
  }

  // EC2 VPC
  setting {
    namespace = "aws:ec2:vpc"
    name      = "VPCId"
    value     = data.terraform_remote_state.account.outputs.vpc_id
  }

  setting {
    namespace = "aws:ec2:vpc"
    name      = "Subnets"
    value     = data.terraform_remote_state.account.outputs.private_subnets
  }

  setting {
    namespace = "aws:ec2:vpc"
    name      = "AssociatePublicIpAddress"
    value     = "false"
  }

  setting {
    namespace = "aws:ec2:vpc"
    name      = "ELBScheme"
    value     = "internal"
  }

  // Cloudwatch
  setting {
    namespace = "aws:elasticbeanstalk:cloudwatch:logs"
    name      = "StreamLogs"
    value     = "true"
  }
}

output "elastic_beanstalk_environment_api_name" {
  value = aws_elastic_beanstalk_environment.api.name
}
