terraform {
  required_version = ">= 0.12"

  backend "s3" {
    bucket  = "cb-roster-terraform"
    key     = "production/terraform.tfstate"
    region  = "us-west-2"
    profile = "twitch-cb-aws"
    encrypt = true
  }
}

locals {
  environment        = "production"
  vpc_id             = "vpc-7851b71e"
  subnet_ids         = ["subnet-6f4a7808", "subnet-45e7ad0c", "subnet-86fe23dd"]
  security_group_ids = ["sg-57251d2c"]
}

module "app" {
  source                           = "../modules/app"
  environment                      = local.environment
  instance_type                    = "c4.xlarge"
  asg_max_size                     = 30
  asg_min_size                     = 8
  deployment_batch_size_percentage = 25

  sandstorm_iam_role_arn  = "arn:aws:iam::734326455073:role/sandstorm/production/templated/role/cb-roster-production"
  hard_delete_user_arn    = "arn:aws:sns:us-west-2:641044725657:notification-user-destroyed"
  s3_assets_iam_role_arn  = module.s3.assets_arn
  s3_uploads_iam_role_arn = module.s3.uploads_arn
  pdms_role_arn           = "arn:aws:iam::125704643346:role/PDMSLambda-CallerRole-13IIND444YKVR"
  pdms_lambda_arn         = "arn:aws:lambda:us-west-2:125704643346:function:PDMSLambda-LambdaFunction-11LXHJVSNJJPY:live"
}

module "rds" {
  source = "../modules/rds"

  environment = local.environment
}

module "redis" {
  source = "../modules/redis"

  environment = local.environment
}

module "s3" {
  source = "../modules/s3"

  environment = local.environment

  uploads_bucket_name = "twitch-team-asset-upload"
}

module "pagerduty" {
  source = "../modules/pagerduty"

  elastic_beanstalk_app_name                = module.app.elastic_beanstalk_app_name
  elastic_beanstalk_environment_api_name    = module.app.elastic_beanstalk_environment_api_name
  elastic_beanstalk_environment_worker_name = module.app.elastic_beanstalk_environment_worker_name
  primary_rds_instance_identifier           = module.rds.db_identifier
  pagerduty_endpoint                        = "https://events.pagerduty.com/integration/be357810973042318b056b3db0c58109/enqueue"
}

module "glue" {
  source = "../modules/glue"

  environment          = local.environment
  db_subnet_group_name = module.rds.db_subnet_group_name
  db_identifier        = module.rds.db_identifier
}

module "production-liveline-vpc-endpoint-v2" {
  source = "../modules/vpc_endpoint"

  vpc_id               = local.vpc_id
  vpc_endpoint_service = "com.amazonaws.vpce.us-west-2.vpce-svc-09e5395f949c1bf56"
  ssl_domain_name      = "main.production.liveline.twitch.a2z.com"
  subnet_ids           = local.subnet_ids
  security_group_ids   = local.security_group_ids
}

module "production-receiver-vpc-endpoint" {
  source = "../modules/vpc_endpoint"

  vpc_id               = local.vpc_id
  vpc_endpoint_service = "com.amazonaws.vpce.us-west-2.vpce-svc-04bef8bdac64fa932"
  ssl_domain_name      = "production.receiver.twitch.a2z.com"
  subnet_ids           = local.subnet_ids
  security_group_ids   = local.security_group_ids
}

module "ldap-vpc-endpoint-a2z" {
  source = "../modules/vpc_endpoint"

  vpc_id               = local.vpc_id
  vpc_endpoint_service = "com.amazonaws.vpce.us-west-2.vpce-svc-0437151f68c61b808"
  ssl_domain_name      = "ldap.twitch.a2z.com"
  subnet_ids           = local.subnet_ids
  security_group_ids   = local.security_group_ids
}

module "rpm-artifactory-vpc-endpoint" {
  source = "../modules/vpc_endpoint"

  vpc_id               = local.vpc_id
  vpc_endpoint_service = "com.amazonaws.vpce.us-west-2.vpce-svc-0ae1454184d60ff60"
  ssl_domain_name      = "rpm.pkgs.xarth.tv"
  subnet_ids           = local.subnet_ids
  security_group_ids   = local.security_group_ids
}

output "s3_kms_key" {
  value = module.glue.s3_kms_key
}

output "s3_output_bucket" {
  value = module.glue.s3_output_bucket
}

output "glue_role" {
  value = module.glue.glue_role
}
