package circuitmetrics

// adapted from https://git-aws.internal.justin.tv/hygienic/taggingcircuitsmetrics/blob/master/tagging.go
// the above is a good implementation but unfortunately comes with a huge pile of hygienic baggage

import (
	"fmt"
	"strings"
	"time"

	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"github.com/cep21/circuit"
)

// the collector should emit metrics every ten seconds
const collectorDelay = 10 * time.Second

type Outcome int

const (
	Success Outcome = iota + 1
	Failure
	Timeout
	BadRequest
	Interrupt
	ShortCircuit
	ConcurrencyLimitReject
)

type AdditionalMetricRollup struct {
	// Metric emitted will be "Rollup_<RollupName>"
	RollupName string

	// Which Outcomes should emit a '1'
	ReportOnes []Outcome

	// Which Outcomes should emit a '0'
	ReportZeroes []Outcome
}

// CommandFactory allows ingesting tagged metrics
type CommandFactory struct {
	Reporter telemetry.SampleReporter

	// AdditionalMetricRollups optionally specifies any additional metrics which are combinations of 1/0 for different call outcomes
	AdditionalMetricRollups []AdditionalMetricRollup
}

func nameTag(s string) map[string]string {
	if len(s) > 64 {
		s = s[:64]
	}

	return map[string]string{"CircuitName": strings.Map(sanitizeRune, s)}
}

func sanitizeRune(r rune) rune {
	switch {
	case 'a' <= r && r <= 'z':
		return r
	case '0' <= r && r <= '9':
		return r
	case 'A' <= r && r <= 'Z':
		return r
	default:
		return '_'
	}
}

func combineDimensions(dims ...map[string]string) map[string]string {
	capacity := 0
	for _, d := range dims {
		capacity += len(d)
	}

	newDims := make(map[string]string, capacity)

	for _, dim := range dims {
		for k, v := range dim {
			newDims[k] = v
		}
	}

	return newDims
}

func reporterWithDimensions(base telemetry.SampleReporter, dims map[string]string) telemetry.SampleReporter {
	allDims := combineDimensions(base.Dimensions, dims)

	base.Dimensions = allDims

	return base
}

// ConcurrencyCollector runs as a background routine to collect concurrency stats
type ConcurrencyCollector struct {
	Reporter telemetry.SampleReporter
	Manager  *circuit.Manager
	Opts     ConcurrencyCollectorOptions

	onClose    chan struct{}
	closedChan chan struct{}
}

// ConcurrencyCollectorOptions allows configuring the ConcurrencyCollector
type ConcurrencyCollectorOptions struct {
	// SkipZeroValueMetrics specifies whether zero-valued IsOpen, Concurrent, or ConcurrentPct metrics are emitted. Defaults to false.
	SkipZeroValueMetrics bool
}

// Collect reports concurrency information for each circuit. It is called on a
// regular cadence from the Start method.
func (c *ConcurrencyCollector) Collect() {
	for _, circ := range c.Manager.AllCircuits() {
		name := circ.Name()
		concurrent := float64(circ.ConcurrentCommands())
		maxConcurrent := float64(circ.Config().Execution.MaxConcurrentRequests)
		concPercent := concurrent / maxConcurrent * 100.0

		reporter := reporterWithDimensions(c.Reporter, combineDimensions(map[string]string{"Producer": "circuit"}, nameTag(name)))

		isOpen := float64(0)
		if circ.IsOpen() {
			isOpen = 1
		}

		if !c.Opts.SkipZeroValueMetrics || isOpen != 0 {
			reporter.Report("IsOpen", isOpen, telemetry.UnitCount)
		}
		if !c.Opts.SkipZeroValueMetrics || concurrent != 0 {
			reporter.Report("Concurrent", concurrent, telemetry.UnitCount)
			reporter.Report("ConcurrentPct", concPercent, telemetry.UnitCount) // ideally the unit would be percent, which would work for CW
		}
	}
}

// Start blocks forever and runs circuit collection every `collectorDelay`
func (c *ConcurrencyCollector) Start() {
	c.onClose = make(chan struct{})
	c.closedChan = make(chan struct{})

	ticker := time.NewTicker(collectorDelay)

	defer func() {
		ticker.Stop()
		close(c.closedChan)
	}()

	for {
		c.Collect()
		select {
		case <-c.onClose:
			return
		case <-ticker.C:
		}
	}
}

// Close stops the processor.
func (c *ConcurrencyCollector) Close() error {
	if c.onClose != nil {
		close(c.onClose)
	}

	<-c.closedChan

	return nil
}

// ConcurrencyCollector returns a ConcurrencyCollector, which can be used to collect stats on each circuit tracked
// by the manager.  You should call `Start` on the returned object and `Close` when you are done collecting metrics.
func (c *CommandFactory) ConcurrencyCollector(manager *circuit.Manager) *ConcurrencyCollector {
	return &ConcurrencyCollector{
		Manager:  manager,
		Reporter: c.Reporter,
	}
}

// CommandProperties creates metrics for a circuit
func (c *CommandFactory) CommandProperties(circuitName string) circuit.Config {
	runner := &runMetricsCollector{
		SampleReporter: reporterWithDimensions(c.Reporter, combineDimensions(map[string]string{"Producer": "circuit.run"}, nameTag(circuitName))),
	}
	if len(c.AdditionalMetricRollups) != 0 {
		additionalMetrics := map[Outcome]map[string]float64{
			Success:                {},
			Failure:                {},
			Timeout:                {},
			BadRequest:             {},
			Interrupt:              {},
			ShortCircuit:           {},
			ConcurrencyLimitReject: {},
		}

		for _, a := range c.AdditionalMetricRollups {
			metric := fmt.Sprintf("Rollup_%s", strings.Map(sanitizeRune, a.RollupName))
			for _, outcome := range a.ReportOnes {
				additionalMetrics[outcome][metric] = 1.0
			}
			for _, outcome := range a.ReportZeroes {
				additionalMetrics[outcome][metric] = 0.0
			}
		}
		runner.additionalMetrics = additionalMetrics
	}

	return circuit.Config{
		Metrics: circuit.MetricsCollectors{
			Run: []circuit.RunMetrics{
				runner,
			},
			Fallback: []circuit.FallbackMetrics{
				&fallbackMetricsCollector{
					SampleReporter: reporterWithDimensions(c.Reporter, combineDimensions(map[string]string{"Producer": "circuit.fallback"}, nameTag(circuitName))),
				},
			},
			// ordinarily, we'd report open/closed circuit stats
			// here, but that's taken care of by the concurrency
			// collector "gauge" style metrics work better when
			// they're periodically reported
			Circuit: []circuit.Metrics{},
		},
	}
}

// runMetricsCollector collects command metrics
type runMetricsCollector struct {
	telemetry.SampleReporter

	// For (optional) additional metrics
	// Maps from outcome to metrics:output values
	additionalMetrics map[Outcome]map[string]float64
}

func (c *runMetricsCollector) reportExtra(outcome Outcome) {
	am, ok := c.additionalMetrics[outcome]
	if !ok {
		return
	}
	for metric, value := range am {
		c.Report(metric, value, telemetry.UnitCount)
	}
}

func (c *runMetricsCollector) Success(now time.Time, duration time.Duration) {
	c.Report("Success", 1.0, telemetry.UnitCount)
	c.ReportDurationSample("Duration", duration)

	c.reportExtra(Success)
}

func (c *runMetricsCollector) ErrFailure(now time.Time, duration time.Duration) {
	c.Report("Success", 0, telemetry.UnitCount)
	c.Report("Error_Failure", 1.0, telemetry.UnitCount)
	c.ReportDurationSample("Duration", duration)

	c.reportExtra(Failure)
}

func (c *runMetricsCollector) ErrTimeout(now time.Time, duration time.Duration) {
	c.Report("Success", 0, telemetry.UnitCount)
	c.Report("Error_Timeout", 1.0, telemetry.UnitCount)
	c.ReportDurationSample("Duration", duration)

	c.reportExtra(Timeout)
}

func (c *runMetricsCollector) ErrBadRequest(now time.Time, duration time.Duration) {
	c.Report("Success", 0, telemetry.UnitCount)
	c.Report("Error_BadRequest", 1.0, telemetry.UnitCount)
	c.ReportDurationSample("Duration", duration)

	c.reportExtra(BadRequest)
}

func (c *runMetricsCollector) ErrInterrupt(now time.Time, duration time.Duration) {
	c.Report("Success", 0, telemetry.UnitCount)
	c.Report("Error_Interrupt", 1.0, telemetry.UnitCount)
	c.ReportDurationSample("Duration", duration)

	c.reportExtra(Interrupt)
}

func (c *runMetricsCollector) ErrShortCircuit(now time.Time) {
	c.Report("Success", 0, telemetry.UnitCount)
	c.Report("Error_ShortCircuit", 1.0, telemetry.UnitCount)

	c.reportExtra(ShortCircuit)
}

func (c *runMetricsCollector) ErrConcurrencyLimitReject(now time.Time) {
	c.Report("Success", 0, telemetry.UnitCount)
	c.Report("Error_ConcurrencyLimitReject", 1.0, telemetry.UnitCount)

	c.reportExtra(ConcurrencyLimitReject)
}

var _ circuit.RunMetrics = new(runMetricsCollector)

// fallbackMetricsCollector collects fallback metrics
type fallbackMetricsCollector struct {
	telemetry.SampleReporter
}

func (c *fallbackMetricsCollector) Success(now time.Time, duration time.Duration) {
	c.Report("Success", 1.0, telemetry.UnitCount)
	c.ReportDurationSample("Duration", duration)
}

func (c *fallbackMetricsCollector) ErrConcurrencyLimitReject(now time.Time) {
	c.Report("Success", 0, telemetry.UnitCount)
	c.Report("Error_ConcurrencyLimitReject", 1.0, telemetry.UnitCount)
}

func (c *fallbackMetricsCollector) ErrFailure(now time.Time, duration time.Duration) {
	c.Report("Success", 0, telemetry.UnitCount)
	c.Report("Error_Failure", 1.0, telemetry.UnitCount)
	c.ReportDurationSample("Duration", duration)
}

var _ circuit.FallbackMetrics = new(fallbackMetricsCollector)
