package sauron

import (
	"context"
	"encoding/json"

	"code.justin.tv/cb/sauron/activity/api"
	"code.justin.tv/foundation/twitchclient"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/lambda"
	"github.com/aws/aws-sdk-go/service/lambda/lambdaiface"
	"github.com/pkg/errors"
)

// Client implements client interfaces required by `edge/graphql`.
type Client interface {
	GetActivities(ctx context.Context, input api.GetInput, opts *twitchclient.ReqOpts) (api.GetOutput, error)
}

// ClientImpl implements Client as the name clearly states.
type ClientImpl struct {
	Lambda      lambdaiface.LambdaAPI
	FunctionARN string
}

// NewClient returns a Sauron client, implementing the NewClient interface that `edge/graphql` requires.
func NewClient(conf twitchclient.ClientConf) (ClientImpl, error) {
	sess, err := session.NewSession(&aws.Config{
		HTTPClient: twitchclient.NewHTTPClient(conf),
		Logger:     conf.Logger,
		Region:     aws.String("us-west-2"),
	})
	if err != nil {
		return ClientImpl{}, err
	}

	return ClientImpl{
		Lambda:      lambda.New(sess),
		FunctionARN: conf.Host,
	}, nil
}

// GetActivities Lambda function
// ======================
//	Staging    | arn:aws:lambda:us-west-2:989470033077:function:cb-sauron-staging-api-get-activities
//	Production | arn:aws:lambda:us-west-2:989470033077:function:cb-sauron-production-api-get-activities

// GetActivities returns a list of Activity records for a given channel ID.
func (c ClientImpl) GetActivities(ctx context.Context, input api.GetInput, opts *twitchclient.ReqOpts) (api.GetOutput, error) {
	payload, err := json.Marshal(input)
	if err != nil {
		return api.GetOutput{}, errors.Wrap(err, "failed to json marshal request input")
	}

	output, err := c.Lambda.InvokeWithContext(ctx, &lambda.InvokeInput{
		FunctionName:   aws.String(c.FunctionARN),
		InvocationType: aws.String("RequestResponse"),
		Payload:        payload,
	})
	if err != nil {
		return api.GetOutput{}, err
	}

	if output.StatusCode != nil && *output.StatusCode != 200 {
		return api.GetOutput{}, errors.Errorf("status code %d: %s", *output.StatusCode, output.Payload)
	}

	// When the function returns an error, the HTTP response status code is still 200 (for the `RequestResponse` type),
	// but `lambda.InvokeOutput.FunctionError` will be populated with the value of the `X-Amz-Function-Error` response header.
	if output.FunctionError != nil {
		// This FunctionError is NOT the error string returned by the function --
		// it merely indicates whether the function returned an error.
		var errorResp api.ErrorResponse

		if err = json.Unmarshal(output.Payload, &errorResp); err != nil {
			return api.GetOutput{}, errors.Wrap(err, "failed to json unmarshal function error payload")
		}

		// Ultimately, the consumer of this client will not be able to get around an error string comparison.
		return api.GetOutput{}, errorResp
	}

	var response api.GetOutput

	if err = json.Unmarshal(output.Payload, &response); err != nil {
		return api.GetOutput{}, errors.Wrap(err, "failed to json unmarshal payload")
	}

	return response, nil
}
