package defs

import (
	"fmt"
	"log"
	"strings"
)

type Field struct {
	Name             string `json:"name"`
	PrettyName       string `json:"pretty_name"`
	ShortName        string `json:"short_name"`
	DDBFieldName     string `json:"ddb_name"`
	SNSName          string `json:"sns_name"`
	EventBusName     string `json:"event_bus_name"`
	NoSNSField       bool   `json:"no_sns_field"`
	Type             string `json:"type"`
	ProtoTypeRaw     string `json:"proto_type"`
	DDBValueRaw      string `json:"ddb_value"`
	PubSubValueRaw   string `json:"pubsub_value"`
	Optional         bool   `json:"optional"`
	Required         bool   `json:"required"`
	FailValidationIf string `json:"fail_validation_if"`
	AnonymousIf      string `json:"anonymous_if"`
	Comment          string `json:"comment"`
	Anonymous        *Field
}

const UserType = "types.User"

func (f *Field) fix(path string, idx int) {
	if f.Name == "" {
		log.Fatalf("No 'name' found in field %d of %s", idx, path)
	}
	if f.Type == "" {
		log.Fatalf("No 'type' found in field %d of %s", idx, path)
	} else if f.Type == "User" {
		f.Type = UserType
	}
	if f.SNSName == "" {
		if f.Type == UserType {
			f.SNSName = f.Name + "_id"
		} else {
			f.SNSName = f.Name
		}
	}
	if f.PrettyName == "" {
		f.PrettyName = makePrettyName(f.Name)
	}
	if f.ShortName == "" {
		f.ShortName = f.PrettyName
	}
}

func (f *Field) fixAnonymous(fields []Field, path string, idx int) {
	if f.AnonymousIf != "" {
		for idx := range fields {
			if fields[idx].Name == f.AnonymousIf {
				f.Anonymous = &fields[idx]
				return
			}
		}
		log.Fatalf("Could not find %s referenced from field %d of %s", f.AnonymousIf, idx, path)
	}
}

func (f *Field) DDBName() string {
	if f.DDBFieldName != "" {
		return f.DDBFieldName
	}

	return f.APIName()
}

func (f *Field) APIName() string {
	if f.Type == UserType {
		return f.Name + "_id"
	}
	return f.Name
}

func (f *Field) APIPrettyName() string {
	if f.Type == UserType {
		return f.PrettyName + "ID"
	}
	return f.PrettyName
}

func (f *Field) APIShortName() string {
	if f.Type == UserType {
		return f.ShortName + "ID"
	}
	return f.ShortName
}

func (f *Field) SNSShortName() string {
	if f.Type == UserType && strings.HasSuffix(f.SNSName, "_id") {
		return f.SNSName[:len(f.SNSName)-3]
	}
	return f.SNSName
}

func (f *Field) SNSPrettyName() string {
	return makePrettyName(f.SNSName)
}

func (f *Field) LocalVariableName() string {
	return variableName(f.PrettyName)
}

func (f *Field) APILocalVariableName() string {
	return variableName(f.APIPrettyName())
}

func (f *Field) ShortLocalVariableName() string {
	return variableName(f.ShortName)
}

func (f *Field) APIType() string {
	if f.Type == UserType {
		return "string"
	}
	return f.Type
}

func (f *Field) ReferenceType() string {
	str := f.Type
	if str[0] == '[' {
		return str
	}
	return "*" + str
}

func (f *Field) APIReferenceType() string {
	str := f.APIType()
	if str[0] == '[' {
		return str
	}
	return "*" + str
}

func (f *Field) ProtoType() string {
	if f.ProtoTypeRaw == "" {
		str := f.APIType()
		if str == "int" {
			str = "int64"
		}
		f.ProtoTypeRaw = str
	}
	return f.ProtoTypeRaw
}

func (f *Field) DDBValue() string {
	if f.DDBValueRaw == "" {
		name := f.APIPrettyName()
		if f.AnonymousIf != "" {
			f.DDBValueRaw = fmt.Sprintf("valueIfNotAnonymous(ddb.%s, ddb.%s)", name, f.Anonymous.APIPrettyName())
		} else {
			f.DDBValueRaw = "ddb." + name
		}
	}
	return f.DDBValueRaw
}

func (f *Field) PubSubValue() string {
	if f.PubSubValueRaw == "" {
		name := f.ShortName
		if f.AnonymousIf != "" {
			name = fmt.Sprintf("data.%s.IfNotAnonymous(data.%s)", name, f.Anonymous.ShortName)
		} else {
			name = "data." + name
			if !f.Optional && f.Type[0] != '[' {
				name = "&" + name
			}
		}
		f.PubSubValueRaw = name
	}
	return f.PubSubValueRaw
}

func (f *Field) HandlerPubSubValue() string {
	value := f.ShortLocalVariableName()
	if f.AnonymousIf != "" {
		value = fmt.Sprintf("%s.IfNotAnonymous(%s)", value, f.Anonymous.ShortLocalVariableName())
	}
	return value
}

func (f *Field) DynamoValue() string {
	name := "data." + f.APIShortName()
	if !f.Optional && f.Type[0] != '[' {
		name = "&" + name
	}
	return name
}

func (f *Field) UsesSharedTypes() bool {
	t := f.Type
	if t[0] == '[' {
		t = t[2:]
	}
	return strings.HasPrefix(t, "types.")
}

func (f *Field) FailValidationCondition() string {
	if f.FailValidationIf == "" {
		if f.Optional {
			f.FailValidationIf = "== nil"
		} else {
			switch f.Type {
			case "types.User":
				fallthrough
			case "string":
				f.FailValidationIf = `== ""`
			case "bool":
				f.FailValidationIf = "== false"
			default:
				f.FailValidationIf = "== 0"
			}
		}
	}
	return f.FailValidationIf
}
