package defs

import (
	"fmt"
	"log"
	"path/filepath"
	"strings"
)

type Handler struct {
	Alert               Alert             `json:"alert"`
	Description         string            `json:"description"`
	Name                string            `json:"name"`
	PrettyName          string            `json:"pretty_name"`
	PackageName         string            `json:"package_name"`
	JenkinsJobName      string            `json:"jenkins_job_name"`
	Fields              []Field           `json:"fields"`
	SharedFields        string            `json:"shared_fields"`
	IgnoreFields        []string          `json:"ignore_fields"`
	IncludeFields       []string          `json:"include_fields"`
	SNS                 SNS               `json:"sns"`
	TerraformModuleName string            `json:"terraform_module_name"`
	Activities          []Activity        `json:"activities"`
	CustomHandler       CustomHandlerInfo `json:"handler"`
	IsEventBus          bool              `json:"is_event_bus"`
	EventBus            EventBus          `json:"event_bus"`
	Category            string
	validFields         []Field
	userFields          []Field
}

func (h *Handler) fix(path, category string) {
	if h.IsEventBus {
		// eventBus-specific fields
		h.EventBus.fix(path)
	} else {
		// Non-eventBus-specific fields (sns)
		h.SNS.fix(path)
		if h.Category == "" {
			h.Category = category
		}
	}

	definitionName := strings.TrimSuffix(filepath.Base(path), filepath.Ext(path))
	if h.Name == "" {
		h.Name = definitionName
	}

	if h.PackageName == "" {
		h.PackageName = strings.ReplaceAll(definitionName, "_", "")
	}
	if h.PrettyName == "" {
		h.PrettyName = makePrettyName(h.Name)
	}

	if h.IsEventBus {
		return
	}

	// All corrections below only apply to SNS messages (ie not event-bus)
	if h.TerraformModuleName == "" {
		h.TerraformModuleName = h.SNS.ARNName
	}

	if h.JenkinsJobName == "" {
		if h.Category != "" {
			h.JenkinsJobName = fmt.Sprintf("handle-%s-%s", strings.ReplaceAll(h.Category, "_", "-"), strings.ReplaceAll(definitionName, "_", "-"))
		} else {
			h.JenkinsJobName = fmt.Sprintf("handle-%s", strings.ReplaceAll(definitionName, "_", "-"))
		}
	}
}

func (h *Handler) fixActivities(path string) {
	if len(h.Activities) == 0 {
		h.Activities = []Activity{{h.Name, h.PrettyName,
			h.Description, h.IgnoreFields, h.IncludeFields,
			h.Fields, nil}}
	} else {
		for idx := range h.Activities {
			h.Activities[idx].fix(h, path, idx)
		}
	}
}

// Fixes properties and fields for eventBus handlers. Requires
// setting Fields first. Ignores non-eventbus handlers.
func (h *Handler) fixEventBus() {
	if !h.IsEventBus {
		return
	}

	for _, field := range h.Fields {
		if field.EventBusName == "" {
			log.Fatalf("Property names for eventBus should go under event_bus_name for field '%s' of handler '%s'", field.Name, h.Name)
		}
	}
}

func (h *Handler) HasAlert() bool {
	return h.Alert.HasAlert()
}

func (h *Handler) LocalVariableName() string {
	return variableName(h.PrettyName)
}

func (h *Handler) PathFragment() string {
	packageName := h.PackageName
	if h.Category == "" {
		return packageName
	}
	return filepath.Join(h.Category, packageName)
}

func (h *Handler) EventName() string {
	prefix := "event"
	if h.IsEventBus {
		prefix = "eventbus"
	}
	if h.Category == "" {
		return fmt.Sprintf("%s.%s", prefix, h.PackageName)
	}
	return fmt.Sprintf("%s.%s.%s", prefix, h.Category, h.PackageName)
}

func (h *Handler) ValidFields() []Field {
	if len(h.validFields) > 0 {
		return h.validFields
	}

	if len(h.IgnoreFields) == 0 && len(h.IncludeFields) == 0 {
		h.validFields = h.Fields
	} else {
		result := filterValidFields(h.Fields, h.IncludeFields, h.IgnoreFields)
		h.validFields = result
	}

	return h.validFields
}

func (h *Handler) UsesSharedTypes() bool {
	for _, field := range h.ValidFields() {
		if field.UsesSharedTypes() {
			return true
		}
	}
	return false
}

func (h *Handler) HasTimestamp() bool {
	return h.SNS.HasTimestamp()
}

func (h *Handler) SNSMessageFields() []SNSField {
	var result []SNSField
	if value, ok := h.SNS.Header["timestamp"]; ok {
		result = append(result, SNSField{"time.Time", value, "Timestamp", false, false, ""})
	} else {
		result = append(result, SNSField{"time.Time", "", "Timestamp", false, false, ""})
	}
	if value, ok := h.SNS.Header["channel_id"]; ok {
		result = append(result, SNSField{"string", value, "ChannelID", false, true, ""})
	} else {
		result = append(result, SNSField{"string", "channel_id", "ChannelID", false, true, ""})
	}
	for _, field := range h.ValidFields() {
		if !field.NoSNSField {
			result = append(result, SNSField{field.APIType(), field.SNSName, makePrettyName(field.SNSName), field.Optional, field.Required || field.FailValidationIf != "", field.FailValidationIf})
		}
	}
	for _, value := range h.SNS.Fields {
		result = append(result, value)
	}
	return result
}

func (h *Handler) Users() []Field {
	if len(h.userFields) == 0 {
		for _, field := range h.ValidFields() {
			if field.Type == UserType {
				h.userFields = append(h.userFields, field)
			}
		}
	}
	return h.userFields
}

func (h *Handler) FirstUser() *Field {
	users := h.Users()
	if len(users) == 0 {
		return nil
	}
	return &users[0]
}

func (h *Handler) FirstUserName() string {
	firstUser := h.FirstUser()
	if firstUser == nil {
		return ""
	}
	return firstUser.ShortName
}

func (h *Handler) HasMultipleActivities() bool {
	return len(h.Activities) > 1
}
