package defs

import (
	"fmt"
	"log"
	"strings"
)

type SNS struct {
	Description               string            `json:"description"`
	Spec                      string            `json:"spec"`
	ARNName                   string            `json:"arn_name"`
	ARN                       string            `json:"arn"`
	StagingARN                string            `json:"staging_arn"`
	ProductionARN             string            `json:"production_arn"`
	Name                      string            `json:"name"`
	ShortName                 string            `json:"short_name"`
	LambdaVariableName        string            `json:"lambda_variable_name"`
	DurationThreshold         int               `json:"duration_threshold"`
	ShortDescription          string            `json:"short_description"`
	ReservedConcurrent        int               `json:"lambda_reserved_concurrent_executions"`
	ReservedConcurrentStaging int               `json:"lambda_reserved_concurrent_executions_staging"`
	Timeout                   int               `json:"timeout"`
	Header                    map[string]string `json:"header"`
	Fields                    []SNSField        `json:"fields"`
	LongestARNName            int
}

func (s *SNS) fix(path string) {
	if s.Spec == "" {
		log.Fatalf("No 'spec' found in 'sns' of %s", path)
	}
	if s.ARNName == "" {
		log.Fatalf("No 'arn_name' found in 'sns' of %s", path)
	}
	if s.StagingARN == "" {
		s.StagingARN = s.ARN
	}
	if s.ProductionARN == "" {
		s.ProductionARN = s.ARN
	}
	if s.StagingARN == "" || s.ProductionARN == "" {
		log.Fatalf("No 'arn' found in 'sns' of %s", path)
	}
	if s.ReservedConcurrent == 0 {
		log.Fatalf("No 'lambda_reserved_concurrent_executions' found in 'sns' of %s (should probably be at least 10)", path)
	}
	if s.ReservedConcurrentStaging == 0 {
		s.ReservedConcurrentStaging = s.ReservedConcurrent
	}
	if s.Name == "" {
		s.Name = strings.ReplaceAll(s.ARNName, "_", "-")
	}
	if s.ShortName == "" {
		parts := strings.Split(s.ARNName, "_")
		s.ShortName = strings.Join(parts[:len(parts)-1], "_")
	}
	if s.LambdaVariableName == "" {
		s.LambdaVariableName = fmt.Sprintf("%s_lambda_name", s.ShortName)
	}
	if s.DurationThreshold == 0 {
		s.DurationThreshold = 5
	}
	if s.ShortDescription == "" {
		s.ShortDescription = strings.ReplaceAll(s.ShortName, "_", " ")
	}
	if s.Description == "" {
		s.Description = fmt.Sprintf("Message is the expected JSON strings from the %s service\n%s", s.Name, s.Spec)
	}
	for idx := range s.Fields {
		s.Fields[idx].fix(path, idx)
	}
}

func (s *SNS) HasTimestamp() bool {
	_, ok := s.Header["timestamp"]
	return ok
}

func (s *SNS) QuotedARNName() string {
	target := s.LongestARNName + 2
	result := fmt.Sprintf(`"%s"`, s.ARNName)
	if len(result) < target {
		result += strings.Repeat(" ", target-len(result))
	}
	return result
}
