package defs

import (
	"encoding/json"
	"fmt"
	"io/ioutil"
	"log"
	"os"
	"path/filepath"
	"sort"
	"strings"
)

type TemplateInfo struct {
	handlers          []Handler
	fieldGroups       []FieldGroup
	alertFieldGroups  []FieldGroup
	SharedFieldGroups int
	AlertIDs          []int
	fieldGroupsSorted bool
	handlersSorted    bool
}

func NewTemplateInfo(folder string) *TemplateInfo {
	templateInfo := &TemplateInfo{}
	templateInfo.parseSharedFields(filepath.Join(folder, "shared_fields"))

	activitiesFolder := filepath.Join(folder, "activities")
	if !isDirectory(activitiesFolder) {
		log.Fatalf("No 'activities' folder found in %s", folder)
	}

	templateInfo.parseDefinitionsInFolder(activitiesFolder, "")
	err := filepath.Walk(activitiesFolder, func(path string, info os.FileInfo, err error) error {
		if err != nil {
			fmt.Printf("got an error walking %q: %v", activitiesFolder, err)
			return err
		}
		if info.IsDir() && info.Name() != "activities" {
			templateInfo.parseDefinitionsInFolder(path, filepath.Base(path))
			return filepath.SkipDir
		}
		return nil
	})
	if err != nil {
		log.Fatalf("Trouble walking folder %q: %v", activitiesFolder, err)
	}
	if len(templateInfo.handlers) == 0 {
		log.Fatalf("No valid template info found in %s", activitiesFolder)
	}
	longestARNName := 0
	for idx := range templateInfo.handlers {
		l := len(templateInfo.handlers[idx].SNS.ARNName)
		if l > longestARNName {
			longestARNName = l
		}
	}
	for idx := range templateInfo.handlers {
		templateInfo.handlers[idx].SNS.LongestARNName = longestARNName
	}

	return templateInfo
}

func (info *TemplateInfo) parseSharedFields(folder string) {
	if isDirectory(folder) {
		files, err := filepath.Glob(filepath.Join(folder, "*.json"))
		if err != nil {
			panic(err)
		}
		for _, file := range files {
			info.parseFieldGroupFile(file)
		}
		info.SharedFieldGroups = len(info.fieldGroups)
	}
}

func (info *TemplateInfo) parseDefinitionsInFolder(folder, category string) {
	files, err := filepath.Glob(filepath.Join(folder, "*.json"))
	if err != nil {
		panic(err)
	}
	for _, file := range files {
		info.parseDefinitionFile(file, category)
	}
}

func (info *TemplateInfo) parseDefinitionFile(path, category string) {
	data, err := ioutil.ReadFile(path)
	if err != nil {
		fmt.Printf("Trouble reading %q: %s", path, err)
		return
	}

	var definition Handler
	err = json.Unmarshal(data, &definition)
	if err != nil {
		fmt.Printf("Trouble parsing %q: %s", path, err)
		return
	}

	definition.fix(path, category)
	definition.Alert.fix(info, path, definition.PrettyName)
	fixFields(definition.Fields, path)
	definition.fixEventBus()
	if definition.SharedFields != "" {
		if len(definition.Fields) != 0 {
			log.Fatalf("You cannot have both 'fields' and 'shared_fields' in %q", path)
		}
		for idx := range info.fieldGroups {
			if idx >= info.SharedFieldGroups {
				break
			}
			f := &info.fieldGroups[idx]
			if f.Name == definition.SharedFields {
				definition.Alert = f.Alert
				definition.Fields = f.Fields
				break
			}
		}
		if len(definition.Fields) == 0 {
			log.Fatalf("Could not find shared fields %q requested by %q", definition.SharedFields, path)
		}
	} else {
		info.fieldGroups = append(info.fieldGroups, FieldGroup{definition.Name, definition.PrettyName, definition.Alert, definition.Fields})
	}
	definition.fixActivities(path)
	info.handlers = append(info.handlers, definition)
}

func (info *TemplateInfo) parseFieldGroupFile(path string) {
	data, err := ioutil.ReadFile(path)
	if err != nil {
		fmt.Printf("Trouble reading %q: %s", path, err)
		return
	}

	var fieldGroup FieldGroup
	err = json.Unmarshal(data, &fieldGroup)
	if err != nil {
		fmt.Printf("Trouble parsing %q: %s", path, err)
		return
	}
	fieldGroup.Name = strings.TrimSuffix(filepath.Base(path), filepath.Ext(path))
	fieldGroup.PrettyName = makePrettyName(fieldGroup.Name)
	fieldGroup.fix(info, path)
	info.fieldGroups = append(info.fieldGroups, fieldGroup)
}

func (info *TemplateInfo) FieldGroups() []FieldGroup {
	if info.fieldGroupsSorted {
		return info.fieldGroups
	}
	sort.SliceStable(info.fieldGroups, func(i, j int) bool {
		if info.fieldGroups[i].Alert.ID > 0 && info.fieldGroups[j].Alert.ID > 0 {
			return info.fieldGroups[i].Alert.ID < info.fieldGroups[j].Alert.ID
		}
		if info.fieldGroups[i].Alert.ID > 0 {
			return true
		}
		if info.fieldGroups[j].Alert.ID > 0 {
			return false
		}
		return info.fieldGroups[i].Name < info.fieldGroups[j].Name
	})
	info.fieldGroupsSorted = true
	return info.fieldGroups
}

func (info *TemplateInfo) AlertFieldGroups() []FieldGroup {
	if len(info.alertFieldGroups) > 0 {
		return info.alertFieldGroups
	}
	for _, fieldGroup := range info.FieldGroups() {
		if fieldGroup.Alert.HasAlert() {
			info.alertFieldGroups = append(info.alertFieldGroups, fieldGroup)
		}
	}
	return info.alertFieldGroups
}

func (info *TemplateInfo) Handlers() []Handler {
	if info.handlersSorted {
		return info.handlers
	}
	sort.SliceStable(info.handlers, func(i, j int) bool {
		if info.handlers[i].Alert.ID > 0 && info.handlers[j].Alert.ID > 0 {
			return info.handlers[i].Alert.ID < info.handlers[j].Alert.ID
		}
		if info.handlers[i].Alert.ID > 0 {
			return true
		}
		if info.handlers[j].Alert.ID > 0 {
			return false
		}
		return info.handlers[i].Name < info.handlers[j].Name
	})
	info.handlersSorted = true
	return info.handlers
}

func (info *TemplateInfo) SNSHandlers() []Handler {
	var handlers []Handler
	for _, handler := range info.Handlers() {
		if !handler.IsEventBus {
			handlers = append(handlers, handler)
		}
	}

	return handlers
}

func (info *TemplateInfo) NextAlertID() int {
	highest := 4
	for _, usedID := range info.AlertIDs {
		if usedID > highest {
			highest = usedID
		}
	}
	return highest + 1
}
