package main

import (
	"flag"
	"fmt"
	"log"
	"os"
	"path/filepath"

	"code.justin.tv/cb/sauron/cmd/codegen/defs"
)

func Usage() {
	_, _ = fmt.Fprintf(os.Stderr, "Usage of codegen:\n")
	_, _ = fmt.Fprintf(os.Stderr, "\tcodegen [flags] definition_directory\n")
	_, _ = fmt.Fprintf(os.Stderr, "Flags:\n")
	flag.PrintDefaults()
}

func main() {
	log.SetFlags(0)
	log.SetPrefix("codegen: ")
	flag.Usage = Usage
	force := flag.Bool("f", false, "force write")
	flag.Parse()

	// We accept either one directory or a list of files. Which do we have?
	args := flag.Args()
	if len(args) == 0 {
		// Default: process all templates in current directory.
		args = []string{"."}
	}
	if len(args) != 1 || !isDirectory(args[0]) {
		flag.Usage()
		os.Exit(2)
	}
	folder := filepath.Clean(args[0])
	folderAbs, err := filepath.Abs(folder)
	if err != nil {
		log.Fatalf("Trouble with %q: %v", folder, err)
	}
	if !isDirectory(folder) {
		log.Printf("%s is not a folder", folder)
		flag.Usage()
		os.Exit(2)
	}

	templateFolder := filepath.Join(folder, "templates")
	if !isDirectory(templateFolder) {
		log.Fatalf("No 'templates' folder found in %s", folder)
	}

	templateInfo := defs.NewTemplateInfo(folder)

	rootPath := filepath.Dir(folderAbs)
	err = filepath.Walk(templateFolder, func(path string, info os.FileInfo, err error) error {
		if err != nil {
			fmt.Printf("Trouble with %q: %v\n", path, err)
		}
		if info == nil || info.IsDir() {
			return nil
		}
		ProcessTemplate(path, rootPath, templateInfo, *force)
		return nil
	})
	if err != nil {
		log.Fatalf("Trouble walking %s", templateFolder)
	}
}

// isDirectory reports whether the named file is a directory.
func isDirectory(name string) bool {
	info, err := os.Stat(name)
	if err != nil {
		return false
	}
	return info.IsDir()
}
