package main

import (
	"bytes"
	"fmt"
	"go/format"
	"io/ioutil"
	"log"
	"os"
	"path/filepath"
	"strings"
	"text/template"
	"time"

	"code.justin.tv/cb/sauron/cmd/codegen/defs"
)

func ProcessTemplate(path string, root string, templateInfo *defs.TemplateInfo, forceWrite bool) {
	data, err := ioutil.ReadFile(path)
	if err != nil {
		log.Fatalf("Trouble reading %q: %s\n", path, err)
		return
	}
	if len(data) == 0 {
		log.Printf("Skipping %q since it is empty\n", path)
		return
	}
	templateData := string(data)
	editable := strings.HasPrefix(templateData, "// editable\n")
	if editable {
		templateData = strings.TrimLeft(templateData[11:], " \t\n")
	}
	funcMap := template.FuncMap{
		// The name "title" is what the function will be called in the template text.
		"add": func(first int, second int) int { return first + second },
	}
	t := template.Must(template.New(path).Funcs(funcMap).Parse(templateData))
	if strings.Contains(path, "{name}") {
		ApplyMultiFileTemplate(path, root, templateInfo, t, forceWrite, editable)
	} else {
		ApplySingleFileTemplate(path, root, templateInfo, t, forceWrite)
	}
}

func ApplyMultiFileTemplate(path string, root string, templateInfo *defs.TemplateInfo, t *template.Template, forceWrite, editable bool) {
	isEventbusPath := strings.Contains(path, "/eventbus/{name}")
	for idx := range templateInfo.Handlers() {
		definition := &templateInfo.Handlers()[idx]

		if (definition.IsEventBus && !isEventbusPath) || (!definition.IsEventBus && isEventbusPath) {
			continue
		}

		destinationPath := strings.Replace(path, "{name}", definition.PathFragment(), 1)
		if editable && !forceWrite {
			outputPath := getTemplateOutputPath(destinationPath, root)
			if oldData, err := ioutil.ReadFile(outputPath); err == nil {
				if len(oldData) > 0 && !strings.HasPrefix(string(oldData), "// Code generated by sauron/cmd/codegen; DO NOT EDIT.") {
					continue
				}
			}
		}
		var buf bytes.Buffer
		err := t.Execute(&buf, definition)
		if err != nil {
			log.Fatalf("Trouble applying %q: %s\n", path, err)
		}
		writeTemplateOutput(path, destinationPath, root, buf, forceWrite, editable)
	}
}

func ApplySingleFileTemplate(path string, root string, templateInfo *defs.TemplateInfo, t *template.Template, forceWrite bool) {
	var buf bytes.Buffer
	err := t.Execute(&buf, templateInfo)
	if err != nil {
		log.Fatalf("Trouble applying %q: %s\n", path, err)
	}
	writeTemplateOutput(path, path, root, buf, forceWrite, false)
}

func writeTemplateOutput(templatePath string, path string, root string, buf bytes.Buffer, forceWrite, editable bool) {
	var output string
	outputPath := getTemplateOutputPath(path, root)
	if filepath.Ext(outputPath) == ".go" {
		content, err := format.Source(buf.Bytes())
		if err != nil {
			log.Fatalf("The go/format step failed for %q: %s\n", path, err)
		}
		output = string(content)

		// Sometimes go/format fails to order the imports correctly so we'll fix that manually
		lines := strings.Split(output, "\n")
		var fixedLines []string
		var simpleImports []string
		var dottedImports []string
		foundImport := false
		for idx, line := range lines {
			if foundImport {
				if line == ")" {
					if len(simpleImports) > 0 {
						fixedLines = append(fixedLines, simpleImports...)
						if len(dottedImports) > 0 {
							fixedLines = append(fixedLines, "")
						}
					}
					if len(dottedImports) > 0 {
						fixedLines = append(fixedLines, dottedImports...)
					}
					fixedLines = append(fixedLines, lines[idx:]...)
					break
				}
				if strings.Contains(line, ".") {
					dottedImports = append(dottedImports, line)
				} else if len(line) > 0 {
					simpleImports = append(simpleImports, line)
				}
			} else {
				if line == "import (" {
					foundImport = true
				}
				fixedLines = append(fixedLines, line)
			}
		}
		output = strings.Join(fixedLines, "\n")
	} else {
		output = buf.String()
	}
	output = strings.TrimLeft(strings.TrimRight(output, "\n\t "), "\n")

	if !forceWrite {
		oldData, err := ioutil.ReadFile(outputPath)
		if err == nil {
			newLines := strings.Split(output, "\n")
			oldLines := strings.Split(string(oldData), "\n")
			for len(oldLines) > 0 && oldLines[0] != newLines[0] {
				oldLines = oldLines[1:]
			}
			oldContent := strings.TrimRight(strings.Join(oldLines, "\n"), "\n\t ")
			if oldContent == output {
				// No change so we can skip it
				return
			}
		}

	}
	output = addHeader(output, templatePath, editable)
	err := ioutil.WriteFile(outputPath, []byte(output), 0644)
	if _, ok := err.(*os.PathError); ok {
		folder := filepath.Dir(outputPath)
		if err = os.MkdirAll(folder, 0755); err == nil {
			err = ioutil.WriteFile(outputPath, []byte(output), 0644)
		}
	}
	if err != nil {
		log.Fatalf("Trouble writing %q: %s\n", outputPath, err)
	} else {
		log.Printf("Updated %s\n", outputPath)
	}
}

const editableHeader = `// Code generated by sauron/cmd/codegen; DO NOT EDIT.
//
// NOTE: You may need to hand edit this file!
//       If so, change the first line from DO NOT EDIT to EDITED (or delete
//       this entire comment block)! After that, the code generator will no
//       longer update this file, so be sure to keep it up to date.
//
// This file was generated by robots at
// %s
// Template path: %s

%s
`

const readOnlyHeader = `// Code generated by sauron/cmd/codegen; DO NOT EDIT.
// This file was generated by robots at
// %s
// Template path: %s

%s
`

func addHeader(content string, templatePath string, editable bool) string {
	var header string
	if editable {
		header = editableHeader
	} else {
		header = readOnlyHeader
	}
	return fmt.Sprintf(header, time.Now(), templatePath, content)
}

func getTemplateOutputPath(templatePath string, root string) string {
	path := filepath.ToSlash(templatePath)
	path = path[strings.Index(path, "/")+1:]
	path = path[strings.Index(path, "/")+1:]
	path = filepath.Join(root, path)
	path = path[:len(path)-5]
	return filepath.Clean(path)
}
