package main

import (
	"code.justin.tv/cb/sauron/internal/clients/pdms"
	"code.justin.tv/cb/sauron/internal/eventbus/follow"
	"code.justin.tv/cb/sauron/internal/eventbus/raiding"
	"code.justin.tv/eventbus/client/subscriber/lambdafunc"
	"code.justin.tv/eventbus/schema/pkg/raid"
	"code.justin.tv/eventbus/schema/pkg/user"
	"context"
	"os"

	"code.justin.tv/cb/sauron/internal/alerts"
	"code.justin.tv/cb/sauron/internal/clients/dynamodb"
	"code.justin.tv/cb/sauron/internal/clients/liveline"
	"code.justin.tv/cb/sauron/internal/clients/pubsub"
	"code.justin.tv/cb/sauron/internal/clients/secrets"
	"code.justin.tv/cb/sauron/internal/clients/stats"
	"code.justin.tv/cb/sauron/internal/clients/users"
	"code.justin.tv/cb/sauron/internal/eventbus/userdestroy"
	"code.justin.tv/cb/sauron/internal/rollbar"
	"code.justin.tv/cb/sauron/internal/util"
	eventbus "code.justin.tv/eventbus/client"
	userfollow "code.justin.tv/eventbus/schema/pkg/user_follow_user"
	"github.com/aws/aws-lambda-go/lambda"
	log "github.com/sirupsen/logrus"
)

func main() {
	env := os.Getenv("ENVIRONMENT")
	if env == "" {
		env = "development"
	}

	secretsManager, err := secrets.NewManager()
	if err != nil {
		log.WithError(err).Error("could not create secrets manager: ", err)
		return
	}

	rollbarToken, err := secretsManager.GetSecret(os.Getenv("ROLLBAR_TOKEN_SECRET_NAME"), os.Getenv("ROLLBAR_TOKEN_SECRET_KEY"))
	if err != nil {
		log.Fatal("sauron: could not retrieve rollbar token: ", err)
	}

	rollbar.SetupLogging(rollbarToken, env)

	statsdClient, err := stats.NewClient(os.Getenv("STATSD_HOST"), env)
	if err != nil {
		log.WithError(err).Fatal("failed to initialize statsd client")
		return
	}

	dynamoCredentials := util.GetCredsForLambda()
	dynamoDBClient, err := dynamodb.NewClient(dynamodb.DynamoClientParams{
		ActivityTable:   os.Getenv("DYNAMODB_TABLE_NAME"),
		AlertPrefsTable: os.Getenv("ALERT_PREFERENCES_TABLE_NAME"),
		RateLimitTable:  os.Getenv("RATE_LIMIT_TABLE_NAME"),
		Region:          os.Getenv("AWS_REGION"),
		Statsd:          statsdClient,
		Credentials:     dynamoCredentials,
	})
	if err != nil {
		log.WithError(err).Fatal("failed to initialize dynamodb client")
		return
	}

	pubsubClient, err := pubsub.NewClient(env, os.Getenv("PUBSUB_HOST"))
	if err != nil {
		log.WithError(err).Fatal("failed to initialize pubsub client")
		return
	}

	usersClient, err := users.NewClient(os.Getenv("USERS_SERVICE_HOST"))
	if err != nil {
		log.WithError(err).Fatal("failed to initialize users service client")
		return
	}

	livelineClient := liveline.NewClient(os.Getenv("LIVELINE_HOST"), statsdClient)

	params := alerts.AlertManagerParams{
		Liveline: livelineClient,
		DynamoDB: dynamoDBClient,
		Pubsub:   pubsubClient,
		Statsd:   statsdClient,
	}

	alertManager := alerts.NewAlertManager(params)

	var pdmsClient = pdms.NewClient(pdms.ClientConfig{
		CallerRoleArn: os.Getenv("PDMS_CALLER_ROLE_ARN"),
		LambdaArn:     os.Getenv("PDMS_LAMBDA_ARN"),
	})

	mux := eventbus.NewMux()

	followHandler := follow.Handler{
		DynamoDB:     dynamoDBClient,
		Pubsub:       pubsubClient,
		Users:        usersClient,
		Statsd:       statsdClient,
		AlertManager: alertManager,
	}
	userfollow.RegisterCreateHandler(mux, followHandler.Handle)

	raidingHandler := raiding.Handler{
		DynamoDB:     dynamoDBClient,
		Pubsub:       pubsubClient,
		Users:        usersClient,
		Statsd:       statsdClient,
		AlertManager: alertManager,
	}
	raid.RegisterRaidUpdateHandler(mux, raidingHandler.Handle)

	userDestroyHandler := userdestroy.Handler{
		DynamoDB:   dynamoDBClient,
		Statsd:     statsdClient,
		PdmsClient: pdmsClient,
	}

	user.RegisterDestroyHandler(mux, userDestroyHandler.Handle)

	mux.HandleDefault(handleDefault)

	lambda.Start(lambdafunc.NewSQS(mux.Dispatcher()))
}

func handleDefault(ctx context.Context, message eventbus.RawMessage) error {
	eventType := "unknown"
	if message.Header != nil && message.Header.EventType != "" {
		eventType = message.Header.EventType
	}

	// To make EventBus migrations tolerable, unhandled events create a warning
	// but are not treated as errors.
	log.Warnf("Unhandled eventbus event received: %s", eventType)
	return nil
}
