package main

import (
	"code.justin.tv/video/autoprof/profs3"
	"fmt"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
	"net/http"
	"os"
	"context"

	"code.justin.tv/cb/sauron/internal/alerts"
	"code.justin.tv/cb/sauron/internal/clients/dynamodb"
	"code.justin.tv/cb/sauron/internal/clients/liveline"
	"code.justin.tv/cb/sauron/internal/clients/pubsub"
	"code.justin.tv/cb/sauron/internal/clients/secrets"
	"code.justin.tv/cb/sauron/internal/clients/stats"
	"code.justin.tv/cb/sauron/internal/clients/users"
	"code.justin.tv/cb/sauron/internal/rollbar"
	"code.justin.tv/cb/sauron/internal/sauronserver"
	"code.justin.tv/cb/sauron/internal/util"
	"code.justin.tv/cb/sauron/rpc/sauron"
	"goji.io/pat"

	log "github.com/sirupsen/logrus"
	goji "goji.io"
)

func main() {
	port := os.Getenv("PORT")
	if port == "" {
		log.Fatal("sauron: no port found in env")
		return
	}

	env := os.Getenv("ENVIRONMENT")
	if env == "" {
		log.Fatal("sauron: no environment found in env")
		return
	}

	secretsManager, err := secrets.NewManager()
	if err != nil {
		log.WithError(err).Error("could not create secrets manager: ", err)
		return
	}

	rollbarToken, err := secretsManager.GetSecret(os.Getenv("ROLLBAR_TOKEN_SECRET_NAME"), os.Getenv("ROLLBAR_TOKEN_SECRET_KEY"))
	if err != nil {
		log.Fatal("sauron: could not retrieve rollbar token: ", err)
	}

	// Set up autoprof
	go func() {
		region := "us-west-2"
		bucket := fmt.Sprintf("autoprof-sauron-%v-%v", env, region)
		log.Printf("Setting up autoprof on bucket %v", bucket)

		sess, _ := session.NewSession(&aws.Config{
			Region: aws.String(region),
		})

		autoprofCollector := profs3.Collector{
			S3:       s3.New(sess),
			S3Bucket: bucket,
			OnError: func(err error) error {
				log.Printf("profs3.Collector.Run err=%q", err)
				return nil
			},
		}

		err = autoprofCollector.Run(context.Background())
		if err != nil {
			log.Warn("failed to run autoprof collector", err.Error())
		}
	}()

	rollbar.SetupLogging(rollbarToken, env)

	statsdClient, err := stats.NewClient(os.Getenv("STATSD_HOST"), env)
	if err != nil {
		log.WithError(err).Fatal("failed to initialize statsd client")
		return
	}

	dynamoDBClient, err := dynamodb.NewClient(dynamodb.DynamoClientParams{
		ActivityTable:   os.Getenv("DYNAMODB_TABLE_NAME"),
		AlertPrefsTable: os.Getenv("ALERT_PREFERENCES_TABLE_NAME"),
		RateLimitTable:  os.Getenv("RATE_LIMIT_TABLE_NAME"),
		Region:          os.Getenv("AWS_REGION"),
		Statsd:          statsdClient,
	})
	if err != nil {
		log.WithError(err).Fatal("failed to initialize dynamodb client")
		return
	}

	livelineClient := liveline.NewClient(os.Getenv("LIVELINE_HOST"), statsdClient)

	pubsubClient, err := pubsub.NewClient(env, os.Getenv("PUBSUB_HOST"))
	if err != nil {
		log.WithError(err).Fatal("failed to initialize pubsub client")
		return
	}

	usersClient, err := users.NewClient(os.Getenv("USERS_SERVICE_HOST"))
	if err != nil {
		log.WithError(err).Fatal("failed to initialize users service client")
		return
	}

	params := alerts.AlertManagerParams{
		Liveline: livelineClient,
		DynamoDB: dynamoDBClient,
		Pubsub:   pubsubClient,
		Statsd:   statsdClient,
		Users:    usersClient,
	}

	alertManager := alerts.NewAlertManager(params)

	server := &sauronserver.Server{
		DynamoDB:     dynamoDBClient,
		Statsd:       statsdClient,
		AlertManager: alertManager,
	}

	twirpHandler := sauron.NewSauronServer(server, nil)

	addr := fmt.Sprintf(":%s", port)

	mux := goji.NewMux()
	mux.Handle(pat.Post(sauron.SauronPathPrefix+"*"), twirpHandler)
	mux.HandleFunc(pat.Get("/health"), util.HealthCheck)

	log.Infof("sauron: server listening on port %s", addr)

	if err := http.ListenAndServe(addr, mux); err != nil {
		log.Error("server error: ", err)
	}
}
