# Activity JSON Format

Most aspects of your activity are defined in the JSON files in this folder. You can either create a JSON file directly
in this folder, or nested one level deep. For instance the [subscription folder](./subscription) contains 3 JSON files
that are all related to subscriptions.

You may want to peruse some of the existing files to get an idea of the variety of configurations that we are about to
cover. But let's start with a fairly simple example and look at [`drop_claim_window_open.json`](./drop_claim_window_open.json)

```json
{
  "package_name": "drops",
  "description": "channel has completed a DropQuest and viewers are now able to claim the drop",
  "fields": [
    {
      "name": "drop_name",
      "type": "string",
      "required": true
    },
    {
      "name": "drop_instance_id",
      "type": "string",
      "required": true
    },
    {
      "name": "quest_name",
      "pretty_name": "DropQuestName",
      "type": "string",
      "required": true
    },
    {
      "name": "claim_duration",
      "pretty_name": "DropClaimDuration",
      "type": "int32",
      "fail_validation_if": "<= 0"
    }
  ],
  "sns": {
    "description": "DropClaimWindowOpenMessage is the expected signature of the events.SNSEntity.Message JSON string from TwitchDropsFulfillment service.\n// https://code.amazon.com/packages/TwitchDropsFulfillmentService/blobs/767c5eba29c241b55763480351a1071918ae834f/--/sns/activity_feed_topic.go#L19-L25",
    "spec": "https://code.amazon.com/packages/TwitchDropsFulfillmentService/blobs/767c5eba29c241b55763480351a1071918ae834f/--/sns/activity_feed_topic.go#L19-L25",
    "arn_name": "drops_activity_feed",
    "production_arn": "arn:aws:sns:us-west-2:049096181636:drops-activity-feed",
    "staging_arn": "arn:aws:sns:us-west-2:920044005461:drops-activity-feed",
    "lambda_reserved_concurrent_executions": 20
  }
}
```

## The `"sns"` section

Let's start with the tricky stuff. As described in [activity.md](../../docs/activity.md), "In order to process events, Sauron
 needs to subscribe to an SNS topic for that event, hook it up to an SQS queue, and invoke a lambda to handle messages from that queue."
Every JSON file in this folder is defining a handler for a single SNS topic. So you will need to understand the SNS message
that you need and add the correct configuration in the `"sns"` section. Let's go through the options used above:

- `description` - This is a comment explaining the event. It will be added as a comment to the `message.go` file in your handler.
- `spec` - A link to file in git that defines the message. Try looking at the one in the [above example](https://code.amazon.com/packages/TwitchDropsFulfillmentService/blobs/767c5eba29c241b55763480351a1071918ae834f/--/sns/activity_feed_topic.go#L19-L25).
- `arn_name` - This will be the name of your SNS module in [main.tf](../../terraform/modules/app/main.tf)
- `arn` - The actual `arn` to use for the message
    - If you have separate `arn`s for production and staging, you can use `production_arn` and `staging_arn` as in the above example.
- `lambda_reserved_concurrent_executions` - This should probably be at least 10. TODO: What is this exactly?

## The `"fields"` section

This example is nice and simple because the author was able to keep a one-to-one mapping between the SNS fields and the
activity fields, but there is still a bit to dig into.

- `name` - The name of the field (snake_case). This will be the name used when we write the data to DynamoDB.
- `pretty_name` (PascalCase) - If you do not specify a pretty name, we will generate it from `name`, so `claim_duration`
will become `ClaimDuration` in the `pretty_name`.
- `type` - This can be anything, but only standard types and `User` work automatically. See the `subscription_custom_message_fragments`
field of [subscriptions/notice.json](./subscriptions/notice.json) for an example of a non-standard type.
    - Users and handled specially. In some places, we transmit a full [`User`](../../types/types.go) structure, and in
    others we transmit only the ID. You'll notice that if the `name` is `user`, some of the generated code will say `user_id`.
- `required` - If `required` is `true`, then we will generate code in your `validate.go` to check that the field is not
empty. For a `string`, this will fail if `field == ""`. For numeric, it will fail on `field == 0`.
- `fail_validation_if` - This is a more explicit form of `required`. In the above example we `"fail_validation_if": "<= 0"`, so if
`DropClaimDuration <= 0`, the validation check will fail and the handler will not continue processing the message.

## The other fields

In this example, we also have two top-level fields:

- `package_name` - By default, the package name will be the same as the name of your file, but sometimes it is convenient
to override this. The `package_name` will also be used at the beginning of log messages.
- `description` - This will be inserted as a comment into the code when a `struct` is defined for your message.

## Adding a field

Just to show what happens, let's add a field to [`drop_claim_window_open.json`](./drop_claim_window_open.json). All you
really need is a name and a type. So just go ahead and add one. You should always add new fields to the end. Now do `make codegen`
and use your favorite `git` client to see what changed.

At the time of this writing, you should see 10 files modified. In this case, the code generator was able to modify every
single file that needed to be changed except for `internal/event/drops/handler_test.go`. Of course, since your new random
field doesn't actually exist in the SNS message, this won't actually work. But if there were another field in the SNS
message that you decided was relevant, this would be just about all it takes.

(Go ahead and use `git` to undo that change.)

## Another Example

Let's have a look now at [raiding.json](./raiding.json):
```json
{
  "alert": {
    "id": 8,
    "name": "RaidData",
    "variable_name": "raid"
  },
  "description": "channel completed raiding another channel",
  "fields": [
    {
      "name": "raider",
      "sns_name": "source_channel_id",
      "type": "User",
      "required": true
    },
    {
      "name": "raiding_viewer_count",
      "sns_name": "viewer_count",
      "short_name": "ViewerCount",
      "type": "int64"
    }
  ],
  "sns": {
    "description": "Message is the expected signature of the raidNotificationMessage JSON string from the autohost service (https://git.xarth.tv/live/autohost/blob/master/internal/logic/raid_logic.go#L61)\n// Insignificant fields are omitted.",
    "spec": "https://git.xarth.tv/live/autohost/blob/aae576128d485f8a14cc846f53e640f651d68b9d/internal/logic/raid_logic.go#L61-L67",
    "arn_name": "autohost_raid_notifications",
    "arn": "arn:aws:sns:us-west-2:447680546588:autohost_production_raid_notifications",
    "lambda_reserved_concurrent_executions": 20,
    "header": {
      "timestamp": "event_time",
      "channel_id": "target_channel_id"
    }
  }
}
```

## The `"alert"` section

You'll need this section if your event needs to be converted into a Spotlight alert. If you're unsure whether or not your
activity should be an alert, ask in #dashboard-feedback for assistance.

- `id` - This is the only required field for an alert. Each alert must have a unique numeric ID. This is used to generate
a protobuf entry.
- `name` - By default, the name of the generated protobuf structure will be the PascalCase version of the name of the JSON
file, plus "Data". In the above example, it would have defaulted to "RaidingData".
- `variable_name` - This is the simple name of the type in protobuf. By default it will be the camelCase version of the name
of the JSON file. In the above example, it would have defaulted to "raiding".

## Fine tuning `"fields"`
- `sns_name` - If the SNS message has a name different from what you'd like to use in the generated code and in Dynamo,
you can pair up the names by specifing the name of the field in the SNS message here.
- `short_name` - In various places in the code, we use the short name. By default, this is the PascalCase version of `name`.

## Alternate header mapping in `"sns"`
- `timestamp` - By default, the timestamp is taken directly from the SQS body. If the SNS message has a timestamp field
of its own, you can specify it here. You have to specify it here even if the name of the SNS field is "timestamp".
- `channel_id` - Most SNS messages we use have a `channel_id` and we just use it. But some put the channel ID in another
field.

# Multiple Activities in One Handler

Every handler has at least one "activity". The activity name is usually the name of the JSON file.

But sometimes you want to take a single SNS message and emit one of several activities. A nice simple example of this is in
[hosting.json](./hosting.json)

```json
{
  "alert": {
    "id": 7,
    "name": "HostData",
    "variable_name": "host"
  },
  "fields": [
    {
      "name": "host",
      "sns_name": "host_channel_id",
      "type": "User",
      "required": true
    },
    {
      "name": "hosting_viewer_count",
      "short_name": "ViewerCount",
      "no_sns_field": true,
      "type": "int",
      "optional": true
    }
  ],
  "sns": {
    "description": "Message is the structure of clue_production_hosting_events\n// See https://git.xarth.tv/chat/tmi/blob/b357611acde86eea75024e5382f140a975f982da/client/sns.go#L36",
    "spec": "https://git.xarth.tv/chat/tmi/blob/d705ae39b9f3ccc23e476747a5b1fb8b0158dfd3/client/sns.go#L35-L44",
    "arn_name": "clue_hosting_events",
    "arn": "arn:aws:sns:us-west-2:447680546588:autohost_production_raid_notifications",
    "lambda_reserved_concurrent_executions": 100,
    "header": {
      "timestamp": "timestamp",
      "channel_id": "target_channel_id"
    },
    "fields": [
      {
        "name": "auto_hosted",
        "type": "bool"
      },
      {
        "name": "hosted_after_raiding",
        "type": "bool"
      },
      {
        "name": "viewer_count",
        "type": "int"
      },
      {
        "name": "viewer_count_revealed",
        "type": "bool"
      }
    ]
  },
  "activities": [
    {
      "name": "auto_host_start",
      "description": "channel gets autohosted"
    },
    {
      "name": "host_start",
      "description": "channel begins hosting another channel"
    }
  ]
}
```

## `"fields"` and `"sns": fields`

It looks like a lot, but it's not too crazy. In the custom [`handler.go`](../../internal/event/hosting/handler.go) , we look at
the `auto_hosted` field from the SNS message and emit either the `auto_host_start` or `host_start` activity. You can see that
there are only 2 regular fields. Both of those get emitted as part of both activity types.

But we have several other SNS fields that we care about that we don't want going into our output fields. That is why we
can also have another `"fields"` section within our `"sns"` section. These fields can also have `"required"` and "`fail_validation_if`"
attributes just like regular fields. By adding these here, our handler will have access to them for custom logic.

There is one other odd capability in use here. Notice the `viewer_count` is listed in both sections. It is slightly
different in the SNS message than what we want in the rest of the code, so we simply mark the one in our regular fields as
`"no_sns_field": true` and add it to our SNS fields. Our handler will handle the actual mapping. In this case, we want
our `viewer_count` to be `optional` but that is not the case in the SNS message.

## The `"activities"` section

When present, the activities listed in this section are used instead of the automatic activity that would have been generated
based on the filename. The `description` at the top-level will be ignored and the descriptions of each activity will be
used instead.

Even though they are not used in this example (look [here](./subscription/notice.json) instead), let's go ahead and cover
a couple of other attributes of the `"activities"` section:

- `ignore_fields` - A list of fields to ignore. They are used by another activity but not this one.
- `include_fields` - Use only these fields.

Note: These are only referring to the top-level `"fields"`, not SNS fields. Remember, you only have a single `handler.go`
for each JSON file so your handler can always see all SNS fields.

# Anonymous Users

In some cases, users have opted to remain anonymous. A good example of this can be seen in [bits_usage.json](./bits_usage.json).
Let's just look at a snippet:
```json
{
  "fields": [
    {
      "name": "bits_amount",
      "sns_name": "amount",
      "short_name": "Amount",
      "type": "int64"
    },
    {
      "name": "bits_anonymous",
      "sns_name": "is_anonymous",
      "short_name": "Anonymous",
      "type": "bool"
    },
    {
      "name": "bits_user",
      "sns_name": "user_id",
      "short_name": "User",
      "type": "User",
      "required": true,
      "anonymous_if": "bits_anonymous",
      "comment": "nil if BitsAnonymous"
    }
  ]
}
```

- `comment` - The comment will be dropped into the code wherever this field is defined
- `anonymous_if` - If the referenced field is true, then this user will be anonymous and will not be sent to pubsub
or Spotlight alerts.

# Shared Fields

Activities let's you convert a single SNS into multiple activities. Sometimes you need to use the same fields for
multiple SNS messages. A good example can be seen in [subscription/fulfillment.json](./subscription/fulfillment.json) and
[subscription/mmg.json](./subscription/mmg.json). The first one is used for the `subscription_gifting_individual` activity
and the second for `subscription_gifting_community`. They come from 2 separate SNS messages, so they need to be in 2
separate handlers, but they share all but one field. Furthermore, we want them to share a protobuf structure.

The answer is to put the fields into the [`shared_fields`](../shared_fields) folder. The JSON files in this folder should
have only the `"alert"` and `"fields"` sections. Then in your main JSON file, just say:
```json
"shared_fields": "subscription_gift"
```

Or whatever the name of your JSON file is in the `shared_fields` folder.

# Adding Special Fields to a Handler

Some handlers need extra parameters to pass to their command. For example, here is a snippet from [fulfillment.json](./subscription/fulfillment.json):

```json
{
  "handler": {
    "imports": ["code.justin.tv/cb/sauron/internal/clients/subscriptions"],
    "params": {
      "Subscriptions": {
        "type": "subscriptions.SubService",
        "value": "subscriptions.NewClient(os.Getenv(\"SUBSCRIPTIONS_HOST\"))"
      }
    },
    "validate_sqs": true
  }
}
```

- `imports` - A list of additional imports
- `params` - A map of additional parameters, where the key is the name and `type` and `value` are required.
    - Pro-Tip: If you need some complex code for the `value`, you can manually add a separate file to the folder with
    your `main.go` that provides a helper function that you can call. See an example [here](../../cmd/event/subscription/notice/new_zuma_client.go).
- `validate_sqs` - Sometimes you want to validate the SQS message before you even unmarshal the SNS message. If so, use this.
You can see an example in [the fulfillment validate.go](../../internal/event/subscription/fulfillment/validate.go).

