package activity

import (
	"context"
	"encoding/base64"
	"time"

	"code.justin.tv/cb/sauron/activity/api"
	"code.justin.tv/cb/sauron/internal/clients/dynamodb"
	"code.justin.tv/cb/sauron/internal/clients/stats"
	log "github.com/sirupsen/logrus"
)

// DynamoDB is the expected interface for the DynamoDB table.
type DynamoDB interface {
	GetActivities(ctx context.Context, channelID string, before time.Time, limit int) ([]dynamodb.Activity, error)
}

// Handler implements github.com/aws/aws-lambda-go/lambda.Handler.
type Handler struct {
	DynamoDB DynamoDB
	Statsd   stats.StatSender
}

// Handle is passed as an argument to github.com/aws/aws-lambda-go/lambda.NewHandler.
func (h Handler) Handle(ctx context.Context, input api.GetInput) (*api.GetOutput, error) {
	if len(input.ChannelID) == 0 {
		return nil, api.ErrInvalidChannelID
	}

	beforeTime, err := decodeCursor(input.ChannelID, input.Cursor)
	if err != nil {
		return nil, api.ErrInvalidCursor
	}

	queriedActivities, err := h.DynamoDB.GetActivities(ctx, input.ChannelID, beforeTime, safeLimit(input.Limit))
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"channel_id":  input.ChannelID,
			"before_time": beforeTime,
		}).Error("get: failed to get activities from dynamodb")

		return nil, err
	}

	outputActivities := make([]api.Activity, len(queriedActivities))

	for idx, queried := range queriedActivities {
		outputActivities[idx] = toAPIActivity(queried)
	}

	return &api.GetOutput{
		Activities: outputActivities,
	}, nil
}

func decodeCursor(channelID string, cursor *string) (time.Time, error) {
	if cursor == nil {
		return time.Now(), nil
	}

	logger := log.WithFields(log.Fields{
		"channel_id": channelID,
		"cursor":     *cursor,
	})

	decoded, err := base64.StdEncoding.DecodeString(*cursor)
	if err != nil {
		logger.WithError(err).Error("get: failed to decode cursor from base64")
		return time.Time{}, err
	}

	if len(string(decoded)) == 0 {
		// This is accepted behavior, though not great.
		// However, it should not be a warning, since it's supported.
		logger.Info("get: found empty cursor when trying to decode")
		return time.Now(), nil
	}

	parsed, err := time.Parse(time.RFC3339Nano, string(decoded))
	if err != nil {
		logger.WithError(err).Error("get: failed to parse decoded cursor to rfc3339")
		return time.Time{}, err
	}

	return parsed, nil
}

func encodeToCursor(ts time.Time) string {
	return base64.StdEncoding.EncodeToString([]byte(ts.Format(time.RFC3339Nano)))
}

const maxLimit = 100

func safeLimit(limit int) int {
	if limit < 1 || limit > maxLimit {
		return maxLimit
	}

	return limit
}

func valueIfNotAnonymous(value *string, anonymous *bool) *string {
	if anonymous != nil && !*anonymous {
		return value
	}
	return nil
}
