package activity_test

import (
	"context"
	"errors"
	"testing"
	"time"

	"code.justin.tv/cb/sauron/activity/api"
	"code.justin.tv/cb/sauron/internal/activity"
	"code.justin.tv/cb/sauron/internal/clients/dynamodb"
)

type mockedDynamoDB struct {
	limit                 int
	returnLen             int
	lastActivityTimestamp time.Time
	err                   error
}

func (m *mockedDynamoDB) GetActivities(ctx context.Context, channelID string, before time.Time, limit int) ([]dynamodb.Activity, error) {
	m.limit = limit

	activities := make([]dynamodb.Activity, m.returnLen)

	if len(activities) > 0 {
		activities[len(activities)-1].Timestamp = m.lastActivityTimestamp
	}

	return activities, m.err
}

func TestInputValidationErrors(t *testing.T) {
	strPtr := func(val string) *string { return &val }
	handler := activity.Handler{}

	tests := []struct {
		input    api.GetInput
		expected error
	}{
		{
			input: api.GetInput{
				ChannelID: "",
				Cursor:    nil,
			},
			expected: api.ErrInvalidChannelID,
		},
		{
			input: api.GetInput{
				ChannelID: "123",
				Cursor:    strPtr("invalid"),
			},
			expected: api.ErrInvalidCursor,
		},
	}

	for _, test := range tests {
		if _, err := handler.Handle(context.Background(), test.input); err != test.expected {
			t.Errorf("expected '%s', got '%s'", test.expected, err)
		}
	}
}

func TestDynamoDBError(t *testing.T) {
	mockedError := errors.New("failed")

	mocked := mockedDynamoDB{
		err: mockedError,
	}

	handler := activity.Handler{
		DynamoDB: &mocked,
	}

	_, err := handler.Handle(context.Background(), api.GetInput{
		ChannelID: "123",
	})
	if err != mockedError {
		t.Errorf("expected '%s', got '%s'", mockedError, err)
	}
}

func TestReturnedActivities(t *testing.T) {
	mocked := mockedDynamoDB{}
	ts := time.Date(2000, time.January, 0, 0, 0, 0, 0, time.UTC)
	expectedCursor := "MTk5OS0xMi0zMVQwMDowMDowMFo="

	handler := activity.Handler{
		DynamoDB: &mocked,
	}

	tests := []struct {
		input             api.GetInput
		dynamoDBReturnLen int
		expectedOutputLen int
	}{
		{
			input: api.GetInput{
				ChannelID: "123",
				Limit:     2,
			},
			dynamoDBReturnLen: 1,
			expectedOutputLen: 1,
		},
		{
			input: api.GetInput{
				ChannelID: "123",
				Limit:     1000,
			},
			dynamoDBReturnLen: 100,
			expectedOutputLen: 100,
		},
	}

	for _, test := range tests {
		mocked.lastActivityTimestamp = ts
		mocked.returnLen = test.dynamoDBReturnLen

		output, err := handler.Handle(context.Background(), test.input)
		if err != nil {
			t.Fatalf("unexpected error: '%s'", err)
		}

		activities := output.Activities

		if len(activities) != test.expectedOutputLen {
			t.Fatalf("expected output length to be %d, got %d", test.expectedOutputLen, len(activities))
		}

		lastCursor := activities[len(activities)-1].Cursor

		if lastCursor != expectedCursor {
			t.Errorf("expected cursor to be '%s', got '%s'", expectedCursor, lastCursor)
		}
	}
}
