package alerts

import (
	"context"
	"errors"
	"strings"
	"time"

	"code.justin.tv/cb/sauron/activity"
	"code.justin.tv/cb/sauron/internal/clients/dynamodb"
	"code.justin.tv/cb/sauron/internal/clients/liveline"
	"code.justin.tv/cb/sauron/internal/clients/pubsub"
	"code.justin.tv/cb/sauron/internal/clients/stats"
	"code.justin.tv/cb/sauron/internal/clients/users"

	schema "code.justin.tv/cb/sauron/activity/pubsub"
	serrors "code.justin.tv/cb/sauron/internal/errors"
	log "github.com/sirupsen/logrus"
)

var (
	// validPrefNames contains the names of alert preferences that can be set.
	validPrefNames = map[string]struct{}{
		"dnd_mode_enabled":        {},
		"hide_follows":            {},
		"hide_hosts":              {},
		"hide_raids":              {},
		"hide_bits":               {},
		"hide_subscriptions":      {},
		"hide_gift_subscriptions": {},
	}

	// validStatusNames contains the names of the alert statuses that can be set.
	validStatusNames = map[string]struct{}{
		string(dynamodb.AlertStatusQueued):   {},
		string(dynamodb.AlertStatusPlaying):  {},
		string(dynamodb.AlertStatusPlayed):   {},
		string(dynamodb.AlertStatusSkipped):  {},
		string(dynamodb.AlertStatusRejected): {},
		string(dynamodb.AlertStatusFailed):   {},
		string(dynamodb.AlertStatusOffline):  {},
		string(dynamodb.AlertStatusPurged):   {},
	}
)

// Manager is the interface for alerts and alert queue management
type Manager interface {
	GetAlertStatus(ctx context.Context, channelID string, activityType string) Status
	FailAlertStatus(ctx context.Context, channelID string, activityID string)

	UpdateAndPublishAlert(ctx context.Context, channelID string, activityID string, newStatus string) (*dynamodb.Activity, error)

	GetAlertPrefs(ctx context.Context, channelID string) (*dynamodb.AlertPreferences, error)
	SetAlertPrefs(ctx context.Context, channelID string, Key string, Val bool) (*dynamodb.AlertPreferences, error)
}

// AlertManager holds functions for managing alert statuses and the state of an alerts queue
type AlertManager struct {
	DynamoDB dynamodb.Database
	Liveline liveline.Liveline
	Pubsub   pubsub.Publisher
	Statsd   stats.StatSender
	Users    users.Users
}

// AlertManagerParams contains the required parameters for creating an alert manager
type AlertManagerParams struct {
	DynamoDB dynamodb.Database
	Liveline liveline.Liveline
	Pubsub   pubsub.Publisher
	Statsd   stats.StatSender
	Users    users.Users
}

// Status contains the status of an alert, and holds the information required to know if we can publish it
type Status struct {
	CanPublish bool
	StatusName dynamodb.AlertStatus
}

// NewAlertManager creates a new instance of an alert manager
func NewAlertManager(params AlertManagerParams) *AlertManager {
	return &AlertManager{
		DynamoDB: params.DynamoDB,
		Liveline: params.Liveline,
		Pubsub:   params.Pubsub,
		Statsd:   params.Statsd,
		Users:    params.Users,
	}
}

// GetAlertStatus returns the proper status for an alert, based on channel conditions. The default status is
// "queued". If a channel has settings to ignore alerts, we set the status to "rejected". If the channel is
// currently offline, we use "offline". Offline is basically equivalent to "played", but we provide a
// distinction in case we ever render a view that listens for offline messages.
func (m *AlertManager) GetAlertStatus(ctx context.Context, channelID string, activityType string) Status {
	start := time.Now()
	defer func() {
		elapsed := time.Since(start)
		m.Statsd.ExecutionTime("alert_manager.get_status", elapsed)
	}()

	canPublishAlert := false

	// If we error here, we continue the request assuming we cannot publish the alert.
	alertPrefs, err := m.DynamoDB.GetAlertPreferences(ctx, channelID)
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"channel_id": channelID,
		}).Error("alert_manager: failed to get preferences")
	} else {
		// cannot publish an alert if dnd mode is on or follow alerts are disabled
		canPublishAlert = !alertPrefs.DNDModeEnabled && !shouldHideAlertByType(alertPrefs, activityType)
	}

	alertStatus := dynamodb.AlertStatusQueued
	if !canPublishAlert {
		alertStatus = dynamodb.AlertStatusRejected
	}

	// Similar to the dynamo call above, we don't return an error here but instead
	// continue assuming that the channel is offline.
	isLive, err := m.Liveline.IsLive(ctx, channelID)
	if err != nil {
		log.WithField("channel_id", channelID).WithError(err).Error("alert_manager: failed to retrieve liveness from liveline")
	}

	if !isLive {
		alertStatus = dynamodb.AlertStatusOffline
	}

	m.Statsd.GoIncrement("alert_manager.get_status.success", 1)
	return Status{
		CanPublish: canPublishAlert,
		StatusName: alertStatus,
	}
}

// FailAlertStatus sets the alert status to failed. This is intended to be called after an event has successfully been
// inserted, but something else failed afterwards. For example, we inserted the event, but we failed to pubsub the
// alert status.
func (m *AlertManager) FailAlertStatus(ctx context.Context, channelID string, activityID string) {
	start := time.Now()
	defer func() {
		elapsed := time.Since(start)
		m.Statsd.ExecutionTime("alert_manager.fail_status", elapsed)
	}()

	_, err := m.DynamoDB.SetAlertStatus(ctx, channelID, activityID, string(dynamodb.AlertStatusFailed))
	if err != nil {
		log.WithFields(log.Fields{
			"channel_id":  channelID,
			"activity_id": activityID,
		}).WithError(err).Error("alert_manager: could not set alert status to failed")
		m.Statsd.GoIncrement("alert_manager.fail_status.error", 1)
		return
	}

	m.Statsd.GoIncrement("alert_manager.fail_status.success", 1)
}

// GetAlertPrefs returns the alert preferences for a channel.
func (m *AlertManager) GetAlertPrefs(ctx context.Context, channelID string) (*dynamodb.AlertPreferences, error) {
	start := time.Now()
	defer func() {
		elapsed := time.Since(start)
		m.Statsd.ExecutionTime("alert_manager.get_prefs", elapsed)
	}()

	if len(channelID) == 0 {
		log.Warning("alert_manager: get preferences request contained no channel id")

		m.Statsd.GoIncrement("alert_manager.get_prefs.error", 1)
		return nil, serrors.ErrMissingChannelID
	}

	alertPrefs, err := m.DynamoDB.GetAlertPreferences(ctx, channelID)
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"channel_id": channelID,
		}).Error("alert_manager: failed to get preferences")

		m.Statsd.GoIncrement("alert_manager.get_prefs.error", 1)
		return nil, err
	}

	m.Statsd.GoIncrement("alert_manager.get_prefs.success", 1)
	return alertPrefs, nil
}

// SetAlertPrefs changes a single alert preference for a channel.
func (m *AlertManager) SetAlertPrefs(ctx context.Context, channelID string, key string, val bool) (*dynamodb.AlertPreferences, error) {
	start := time.Now()
	defer func() {
		elapsed := time.Since(start)
		m.Statsd.ExecutionTime("alert_manager.set_prefs", elapsed)
	}()

	if len(channelID) == 0 {
		log.Warning("alert_manager: set preferences request contained no channel id")

		m.Statsd.GoIncrement("alert_manager.set_prefs.error", 1)
		return nil, serrors.ErrMissingChannelID
	}

	if len(key) == 0 {
		log.WithField("channel_id", channelID).Error("alert_manager: set preference request contained no preferences")

		m.Statsd.GoIncrement("alert_manager.set_prefs.error", 1)
		return nil, serrors.ErrInvalidPreferenceKey
	}

	formattedKey := strings.ToLower(key)

	if _, ok := validPrefNames[formattedKey]; !ok {
		log.WithFields(log.Fields{
			"channel_id": channelID,
			"key":        formattedKey,
		}).Error("alert_manager: unknown preference provided")

		m.Statsd.GoIncrement("alert_manager.set_prefs.error", 1)
		return nil, serrors.ErrInvalidPreferenceKey
	}

	// If we are setting dnd mode to on, we need to remove all items in the alert queue
	if (formattedKey == "dnd_mode_enabled") && val {
		deleteStart := time.Now()
		err := m.DynamoDB.DeleteAlertQueue(ctx, channelID)
		if err != nil {
			log.WithField("channel_id", channelID).WithError(err).Error("alert_manager: failed to remove items from alert queue")

			m.Statsd.GoIncrement("alert_manager.set_prefs.error", 1)
			m.Statsd.GoExecutionTime("alert_manager.delete_queue.error", time.Since(deleteStart))
			return nil, err
		}

		m.Statsd.GoExecutionTime("alert_manager.delete_queue.success", time.Since(deleteStart))
		err = m.Pubsub.PublishAlert(ctx, channelID, pubsub.Alert{AlertStatus: string(dynamodb.AlertStatusPurged)})
		if err != nil {
			log.WithField("channel_id", channelID).WithError(err).Error("alert_manager: failed to publish queue purging")
		}
	}

	alertPrefs, err := m.DynamoDB.SetAlertPreferences(ctx, channelID, formattedKey, val)
	if err != nil {
		log.WithField("channel_id", channelID).WithError(err).Error("alert_manager: failed to write setting to dynamo db")

		m.Statsd.GoIncrement("alert_manager.set_prefs.error", 1)
		return nil, err
	}

	m.Statsd.GoIncrement("alert_manager.set_prefs.success", 1)
	return alertPrefs, nil
}

// UpdateAndPublishAlert changes the alert status of an activity, and then sends a message to the alerts
// pubsub topic to notify listeners that the status has changed.
func (m *AlertManager) UpdateAndPublishAlert(ctx context.Context, channelID string, activityID string, newStatus string) (*dynamodb.Activity, error) {
	start := time.Now()
	defer func() {
		elapsed := time.Since(start)
		m.Statsd.ExecutionTime("alert_manager.set_and_publish_status", elapsed)
	}()

	if len(channelID) == 0 {
		log.Warning("alert_manager: update status request contained no channel id")

		m.Statsd.GoIncrement("alert_manager.set_and_publish_status.error", 1)
		return nil, serrors.ErrMissingChannelID
	}

	if len(activityID) == 0 {
		log.Warning("alert_manager: update status request contained no activity id")

		m.Statsd.GoIncrement("alert_manager.set_and_publish_status.error", 1)
		return nil, serrors.ErrMissingActivityID
	}

	if _, ok := validStatusNames[newStatus]; !ok {
		log.WithFields(log.Fields{
			"channel_id": channelID,
			"status":     newStatus,
		}).Error("alert_manager: unknown alert status provided")

		m.Statsd.GoIncrement("alert_manager.set_and_publish_status.error", 1)
		return nil, serrors.ErrInvalidAlertStatus
	}

	activity, err := m.DynamoDB.SetAlertStatus(ctx, channelID, activityID, newStatus)
	if err != nil {
		m.Statsd.GoIncrement("alert_manager.set_and_publish_status.error", 1)
		return nil, err
	}

	pubsubData, err := m.convertActivityToPubsub(ctx, activity)
	if err != nil {
		m.Statsd.GoIncrement("alert_manager.set_and_publish_status.error", 1)
		return nil, err
	}

	err = m.Pubsub.PublishAlert(ctx, channelID, pubsub.Alert{
		ActivityID:   activityID,
		ActivityType: activity.Type,
		AlertStatus:  newStatus,
		AlertData:    pubsubData,
	})

	if err != nil {
		m.Statsd.GoIncrement("alert_manager.set_and_publish_status.error", 1)
		return nil, err
	}

	m.Statsd.GoIncrement("alert_manager.set_and_publish_status.success", 1)
	return activity, nil
}

// convertActivityToPubsub takes a dynamodb.Activity struct and converts it to whats expected by pubsub. This mostly entails
// getting user objects, and ensuring that all required data is present
func (m *AlertManager) convertActivityToPubsub(ctx context.Context, currActivity *dynamodb.Activity) (*schema.Activity, error) {
	pubsubActivity := &schema.Activity{
		ID:        currActivity.ID,
		Type:      currActivity.Type,
		Timestamp: currActivity.Timestamp,
	}

	switch currActivity.Type {
	case activity.TypeFollow:
		if currActivity.FollowerID != nil {
			follower, err := m.Users.GetUser(ctx, *currActivity.FollowerID)
			if err != nil {
				return nil, err
			}
			pubsubActivity.Follower = &follower
		}
	case activity.TypeBitsUsage:
		pubsubActivity.BitsAmount = currActivity.BitsAmount
		pubsubActivity.BitsAnonymous = currActivity.BitsAnonymous
		if currActivity.BitsUserID != nil {
			bitsUser, err := m.Users.GetUser(ctx, *currActivity.BitsUserID)
			if err != nil {
				return nil, err
			}
			pubsubActivity.BitsUser = &bitsUser
		}
	case activity.TypeHostStart, activity.TypeAutoHostStart:
		pubsubActivity.HostingViewerCount = currActivity.HostingViewerCount
		if currActivity.HostID != nil {
			hoster, err := m.Users.GetUser(ctx, *currActivity.HostID)
			if err != nil {
				return nil, err
			}
			pubsubActivity.Host = &hoster
		}
	case activity.TypeRaiding:
		pubsubActivity.RaidingViewerCount = currActivity.RaidingViewerCount
		if currActivity.RaiderID != nil {
			raider, err := m.Users.GetUser(ctx, *currActivity.RaiderID)
			if err != nil {
				return nil, err
			}
			pubsubActivity.Raider = &raider
		}
	case activity.TypeSubscriptionGiftingIndividual:
		pubsubActivity.SubscriptionGiftAnonymous = currActivity.SubscriptionGiftAnonymous
		pubsubActivity.SubscriptionGiftQuantity = currActivity.SubscriptionGiftQuantity
		pubsubActivity.SubscriptionGiftTier = currActivity.SubscriptionGiftTier

		if currActivity.SubscriptionGiftAnonymous != nil && !*currActivity.SubscriptionGiftAnonymous && currActivity.SubscriptionGifterID != nil {
			gifter, err := m.Users.GetUser(ctx, *currActivity.SubscriptionGifterID)
			if err != nil {
				return nil, err
			}
			pubsubActivity.SubscriptionGifter = &gifter
		}

		if currActivity.SubscriptionGiftRecipientID != nil {
			recipient, err := m.Users.GetUser(ctx, *currActivity.SubscriptionGiftRecipientID)
			if err != nil {
				return nil, err
			}
			pubsubActivity.SubscriptionGiftRecipient = &recipient
		}
	case activity.TypeSubscriptionGiftingCommunity:
		pubsubActivity.SubscriptionGiftAnonymous = currActivity.SubscriptionGiftAnonymous
		pubsubActivity.SubscriptionGiftQuantity = currActivity.SubscriptionGiftQuantity
		pubsubActivity.SubscriptionGiftTier = currActivity.SubscriptionGiftTier

		if currActivity.SubscriptionGiftAnonymous != nil && !*currActivity.SubscriptionGiftAnonymous && currActivity.SubscriptionGifterID != nil {
			gifter, err := m.Users.GetUser(ctx, *currActivity.SubscriptionGifterID)
			if err != nil {
				return nil, err
			}
			pubsubActivity.SubscriptionGifter = &gifter
		}
	case activity.TypePrimeResubscriptionSharing,
		activity.TypePrimeSubscription,
		activity.TypeResubscriptionSharing,
		activity.TypeSubscription:
		pubsubActivity.SubscriptionTier = currActivity.SubscriptionTier
		pubsubActivity.SubscriptionCumulativeTenureMonths = currActivity.SubscriptionCumulativeTenureMonths
		pubsubActivity.SubscriptionCustomMessageText = currActivity.SubscriptionCustomMessageText

		if currActivity.SubscriberID != nil {
			subscriber, err := m.Users.GetUser(ctx, *currActivity.SubscriberID)
			if err != nil {
				return nil, err
			}
			pubsubActivity.Subscriber = &subscriber
		}

		pubsubActivity.SubscriptionCustomMessageFragments = currActivity.SubscriptionCustomMessageFragments

	default:
		return nil, errors.New("alert_manager: unknown activity type found when trying to convert to pubsub")
	}

	return pubsubActivity, nil
}

// returns true if we should hide the alert by type, false otherwise
func shouldHideAlertByType(alertPrefs *dynamodb.AlertPreferences, activityType string) bool {
	switch activityType {
	case activity.TypeFollow:
		return alertPrefs.HideFollows
	case activity.TypeBitsUsage:
		return alertPrefs.HideBits
	case activity.TypeHostStart, activity.TypeAutoHostStart:
		return alertPrefs.HideHosts
	case activity.TypeRaiding:
		return alertPrefs.HideRaids
	case activity.TypeSubscriptionGiftingIndividual, activity.TypeSubscriptionGiftingCommunity:
		return alertPrefs.HideGiftSubscriptions
	case activity.TypeSubscription, activity.TypeResubscriptionSharing, activity.TypePrimeResubscriptionSharing, activity.TypePrimeSubscription:
		return alertPrefs.HideSubscriptions
	default:
		return true // default to not showing alert if we don't know the event type
	}
}
