package alerts_test

import (
	"context"
	"errors"
	"testing"
	"time"

	"code.justin.tv/cb/sauron/types"

	"code.justin.tv/cb/sauron/activity"
	"code.justin.tv/cb/sauron/internal/alerts"
	"code.justin.tv/cb/sauron/internal/clients/dynamodb"
	"code.justin.tv/cb/sauron/internal/mocks"

	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

type mockManagerParams struct {
	DynamoDB *mocks.Database
	Pubsub   *mocks.Publisher
	Liveline *mocks.Liveline
	Statsd   *mocks.StatSender
	Users    *mocks.Users
}

func managerMocks() mockManagerParams {
	return mockManagerParams{
		DynamoDB: &mocks.Database{},
		Pubsub:   &mocks.Publisher{},
		Liveline: &mocks.Liveline{},
		Statsd:   &mocks.StatSender{},
		Users:    &mocks.Users{},
	}
}

func mockManager(params mockManagerParams) *alerts.AlertManager {
	return &alerts.AlertManager{
		DynamoDB: params.DynamoDB,
		Pubsub:   params.Pubsub,
		Liveline: params.Liveline,
		Statsd:   params.Statsd,
		Users:    params.Users,
	}
}

const (
	channelID  = "111111"
	newStatus  = "played"
	activityID = "activity1"
)

func TestGetAlertStatus(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := managerMocks()
	manager := mockManager(mocks)
	ctx := context.Background()
	mocks.Statsd.On("ExecutionTime", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When given a channel id and event type", t, func() {
		Convey("CanPublish should be false when dnd is enabled", func() {
			prefs := &dynamodb.AlertPreferences{
				ChannelID:      channelID,
				DNDModeEnabled: true,
			}
			eventType := activity.TypeFollow

			mocks.DynamoDB.On("GetAlertPreferences", ctx, channelID).Once().Return(prefs, nil)
			mocks.Liveline.On("IsLive", ctx, channelID).Once().Return(true, nil)

			result := manager.GetAlertStatus(ctx, channelID, eventType)
			So(result.CanPublish, ShouldBeFalse)
			So(result.StatusName, ShouldEqual, dynamodb.AlertStatusRejected)
		})

		Convey("CanPublish should be false when dnd is disabled, but HideFollows is enabled, and the event type is follow", func() {
			prefs := &dynamodb.AlertPreferences{
				ChannelID:      channelID,
				DNDModeEnabled: false,
				HideFollows:    true,
			}
			eventType := activity.TypeFollow

			mocks.DynamoDB.On("GetAlertPreferences", ctx, channelID).Once().Return(prefs, nil)
			mocks.Liveline.On("IsLive", ctx, channelID).Once().Return(true, nil)

			result := manager.GetAlertStatus(ctx, channelID, eventType)
			So(result.CanPublish, ShouldBeFalse)
			So(result.StatusName, ShouldEqual, dynamodb.AlertStatusRejected)
		})

		Convey("CanPublish should be false when dnd is disabled, but HideBits is enabled, and the event type is bitsusage", func() {
			prefs := &dynamodb.AlertPreferences{
				ChannelID:      channelID,
				DNDModeEnabled: false,
				HideBits:       true,
			}
			eventType := activity.TypeBitsUsage

			mocks.DynamoDB.On("GetAlertPreferences", ctx, channelID).Once().Return(prefs, nil)
			mocks.Liveline.On("IsLive", ctx, channelID).Once().Return(true, nil)

			result := manager.GetAlertStatus(ctx, channelID, eventType)
			So(result.CanPublish, ShouldBeFalse)
			So(result.StatusName, ShouldEqual, dynamodb.AlertStatusRejected)
		})

		Convey("CanPublish should be false when dnd is disabled, but HideRaids is enabled, and the event type is raiding", func() {
			prefs := &dynamodb.AlertPreferences{
				ChannelID:      channelID,
				DNDModeEnabled: false,
				HideRaids:      true,
			}
			eventType := activity.TypeRaiding

			mocks.DynamoDB.On("GetAlertPreferences", ctx, channelID).Once().Return(prefs, nil)
			mocks.Liveline.On("IsLive", ctx, channelID).Once().Return(true, nil)

			result := manager.GetAlertStatus(ctx, channelID, eventType)
			So(result.CanPublish, ShouldBeFalse)
			So(result.StatusName, ShouldEqual, dynamodb.AlertStatusRejected)
		})

		Convey("CanPublish should be false when dnd is disabled, but HideHosts is enabled, and the event type is hosting", func() {
			prefs := &dynamodb.AlertPreferences{
				ChannelID:      channelID,
				DNDModeEnabled: false,
				HideHosts:      true,
			}
			eventType := activity.TypeHostStart

			mocks.DynamoDB.On("GetAlertPreferences", ctx, channelID).Once().Return(prefs, nil)
			mocks.Liveline.On("IsLive", ctx, channelID).Once().Return(true, nil)

			result := manager.GetAlertStatus(ctx, channelID, eventType)
			So(result.CanPublish, ShouldBeFalse)
			So(result.StatusName, ShouldEqual, dynamodb.AlertStatusRejected)
		})

		Convey("CanPublish should be false when dnd is disabled, but HideHosts is enabled, and the event type is auto hosting", func() {
			prefs := &dynamodb.AlertPreferences{
				ChannelID:      channelID,
				DNDModeEnabled: false,
				HideHosts:      true,
			}
			eventType := activity.TypeAutoHostStart

			mocks.DynamoDB.On("GetAlertPreferences", ctx, channelID).Once().Return(prefs, nil)
			mocks.Liveline.On("IsLive", ctx, channelID).Once().Return(true, nil)

			result := manager.GetAlertStatus(ctx, channelID, eventType)
			So(result.CanPublish, ShouldBeFalse)
			So(result.StatusName, ShouldEqual, dynamodb.AlertStatusRejected)
		})

		Convey("CanPublish should be false when dnd is disabled, but HideSubscriptions is enabled", func() {
			prefs := &dynamodb.AlertPreferences{
				ChannelID:         channelID,
				DNDModeEnabled:    false,
				HideSubscriptions: true,
			}

			mocks.DynamoDB.On("GetAlertPreferences", ctx, channelID).Once().Return(prefs, nil)
			mocks.Liveline.On("IsLive", ctx, channelID).Once().Return(true, nil)

			Convey("When the event type is subscription", func() {
				eventType := activity.TypeSubscription
				result := manager.GetAlertStatus(ctx, channelID, eventType)

				So(result.CanPublish, ShouldBeFalse)
				So(result.StatusName, ShouldEqual, dynamodb.AlertStatusRejected)
			})

			Convey("When the event type is resubscription", func() {
				eventType := activity.TypeResubscriptionSharing
				result := manager.GetAlertStatus(ctx, channelID, eventType)

				So(result.CanPublish, ShouldBeFalse)
				So(result.StatusName, ShouldEqual, dynamodb.AlertStatusRejected)
			})

			Convey("When the event type is prime subscription", func() {
				eventType := activity.TypePrimeSubscription
				result := manager.GetAlertStatus(ctx, channelID, eventType)

				So(result.CanPublish, ShouldBeFalse)
				So(result.StatusName, ShouldEqual, dynamodb.AlertStatusRejected)
			})

			Convey("When the event type is prime resubscription", func() {
				eventType := activity.TypePrimeResubscriptionSharing
				result := manager.GetAlertStatus(ctx, channelID, eventType)

				So(result.CanPublish, ShouldBeFalse)
				So(result.StatusName, ShouldEqual, dynamodb.AlertStatusRejected)
			})
		})

		Convey("CanPublish should be false when dnd is disabled, but HideGiftSubscriptions is enabled", func() {
			prefs := &dynamodb.AlertPreferences{
				ChannelID:             channelID,
				DNDModeEnabled:        false,
				HideGiftSubscriptions: true,
			}

			mocks.DynamoDB.On("GetAlertPreferences", ctx, channelID).Once().Return(prefs, nil)
			mocks.Liveline.On("IsLive", ctx, channelID).Once().Return(true, nil)

			Convey("When the event type is individual sub gift", func() {
				eventType := activity.TypeSubscriptionGiftingIndividual
				result := manager.GetAlertStatus(ctx, channelID, eventType)

				So(result.CanPublish, ShouldBeFalse)
				So(result.StatusName, ShouldEqual, dynamodb.AlertStatusRejected)
			})

			Convey("When the event type is community sub gift", func() {
				eventType := activity.TypeSubscriptionGiftingCommunity
				result := manager.GetAlertStatus(ctx, channelID, eventType)

				So(result.CanPublish, ShouldBeFalse)
				So(result.StatusName, ShouldEqual, dynamodb.AlertStatusRejected)
			})
		})

		Convey("CanPublish should be true when dnd is disabled, but some subscription types are enabled", func() {
			prefs := &dynamodb.AlertPreferences{
				ChannelID:             channelID,
				DNDModeEnabled:        false,
				HideGiftSubscriptions: true,
			}

			mocks.DynamoDB.On("GetAlertPreferences", ctx, channelID).Once().Return(prefs, nil)
			mocks.Liveline.On("IsLive", ctx, channelID).Once().Return(true, nil)

			Convey("When the event type is prime sub", func() {
				eventType := activity.TypePrimeSubscription
				result := manager.GetAlertStatus(ctx, channelID, eventType)

				So(result.CanPublish, ShouldBeTrue)
				So(result.StatusName, ShouldEqual, dynamodb.AlertStatusQueued)
			})

			Convey("When the event type is a regular subscription", func() {
				eventType := activity.TypeSubscription
				result := manager.GetAlertStatus(ctx, channelID, eventType)

				So(result.CanPublish, ShouldBeTrue)
				So(result.StatusName, ShouldEqual, dynamodb.AlertStatusQueued)
			})
		})

		Convey("The alert status should be queued when the channel is live and we can publish", func() {
			prefs := &dynamodb.AlertPreferences{
				ChannelID:      channelID,
				DNDModeEnabled: false,
			}
			eventType := activity.TypeFollow

			mocks.DynamoDB.On("GetAlertPreferences", ctx, channelID).Once().Return(prefs, nil)
			mocks.Liveline.On("IsLive", ctx, channelID).Once().Return(true, nil)

			result := manager.GetAlertStatus(ctx, channelID, eventType)
			So(result.CanPublish, ShouldBeTrue)
			So(result.StatusName, ShouldEqual, dynamodb.AlertStatusQueued)
		})

		Convey("The alert status should be rejected when the channel is live, but we can't publish", func() {
			prefs := &dynamodb.AlertPreferences{
				ChannelID:      channelID,
				DNDModeEnabled: true,
			}
			eventType := activity.TypeFollow

			mocks.DynamoDB.On("GetAlertPreferences", ctx, channelID).Once().Return(prefs, nil)
			mocks.Liveline.On("IsLive", ctx, channelID).Once().Return(true, nil)

			result := manager.GetAlertStatus(ctx, channelID, eventType)
			So(result.CanPublish, ShouldBeFalse)
			So(result.StatusName, ShouldEqual, dynamodb.AlertStatusRejected)
		})

		Convey("The alert status should be offline when the channel is not live, regardless of publish status", func() {
			prefs := &dynamodb.AlertPreferences{
				ChannelID:      channelID,
				DNDModeEnabled: true,
			}
			eventType := activity.TypeFollow

			mocks.DynamoDB.On("GetAlertPreferences", ctx, channelID).Once().Return(prefs, nil)
			mocks.Liveline.On("IsLive", ctx, channelID).Once().Return(false, nil)

			result := manager.GetAlertStatus(ctx, channelID, eventType)
			So(result.CanPublish, ShouldBeFalse)
			So(result.StatusName, ShouldEqual, dynamodb.AlertStatusOffline)
		})
	})
}

func TestFailAlertStatus(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := managerMocks()
	manager := mockManager(mocks)
	ctx := context.Background()

	mocks.Statsd.On("ExecutionTime", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When calling FailAlertStatus", t, func() {
		Convey("It should call dynamodb", func() {
			mocks.DynamoDB.On("SetAlertStatus", ctx, channelID, activityID, string(dynamodb.AlertStatusFailed)).Once().Return(&dynamodb.Activity{}, nil)
			manager.FailAlertStatus(ctx, channelID, activityID)
			mocks.DynamoDB.AssertExpectations(t)
		})
	})
}

func TestGetAlertPrefs(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := managerMocks()
	manager := mockManager(mocks)
	ctx := context.Background()

	mocks.Statsd.On("ExecutionTime", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When calling GetAlertPrefs", t, func() {
		Convey("When the channel id is empty", func() {
			Convey("It should return an error", func() {
				channelID := ""
				mocks.DynamoDB.AssertNotCalled(t, "GetAlertPreferences", ctx, channelID)

				result, err := manager.GetAlertPrefs(ctx, channelID)
				So(result, ShouldBeNil)
				So(err, ShouldNotBeNil)
				mocks.DynamoDB.AssertExpectations(t)
			})
		})

		Convey("When the call to dynamo fails", func() {
			Convey("It should return an error", func() {
				mocks.DynamoDB.On("GetAlertPreferences", ctx, channelID).Once().Return(nil, errors.New("dynamo error"))

				result, err := manager.GetAlertPrefs(ctx, channelID)
				So(result, ShouldBeNil)
				So(err, ShouldNotBeNil)
				mocks.DynamoDB.AssertExpectations(t)
			})
		})

		Convey("When the call to dynamo succeeds", func() {
			Convey("It should return a valid response", func() {
				resp := &dynamodb.AlertPreferences{
					ChannelID:      channelID,
					DNDModeEnabled: true,
					HideFollows:    true,
					HideRaids:      false,
				}
				mocks.DynamoDB.On("GetAlertPreferences", ctx, channelID).Once().Return(resp, nil)

				result, err := manager.GetAlertPrefs(ctx, channelID)
				So(result, ShouldNotBeNil)
				So(err, ShouldBeNil)
				So(result.ChannelID, ShouldEqual, channelID)
				So(result.DNDModeEnabled, ShouldEqual, true)
				So(result.HideFollows, ShouldEqual, true)
				So(result.HideRaids, ShouldEqual, false)
				So(result.HideHosts, ShouldEqual, false)
				mocks.DynamoDB.AssertExpectations(t)
			})
		})
	})
}

func TestSetAlertPrefs(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := managerMocks()
	manager := mockManager(mocks)
	ctx := context.Background()

	mocks.Statsd.On("ExecutionTime", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When calling SetAlertPrefs", t, func() {
		Convey("When the channel id is empty", func() {
			Convey("It should return an error", func() {
				channelID := ""
				key := "prefkey"
				val := true
				mocks.DynamoDB.AssertNotCalled(t, "SetAlertPreferences", ctx, channelID, key, val)

				_, err := manager.SetAlertPrefs(ctx, channelID, key, val)
				So(err, ShouldNotBeNil)
				mocks.DynamoDB.AssertExpectations(t)
			})
		})

		Convey("When the preference key is empty", func() {
			Convey("It should return an error", func() {
				key := ""
				val := true
				mocks.DynamoDB.AssertNotCalled(t, "SetAlertPreferences", ctx, channelID, key, val)

				_, err := manager.SetAlertPrefs(ctx, channelID, key, val)
				So(err, ShouldNotBeNil)
				mocks.DynamoDB.AssertExpectations(t)
			})
		})

		Convey("When the preference key is not valid", func() {
			Convey("It should return an error", func() {
				key := "invalidkey"
				val := true
				mocks.DynamoDB.AssertNotCalled(t, "SetAlertPreferences", ctx, channelID, key, val)

				_, err := manager.SetAlertPrefs(ctx, channelID, key, val)
				So(err, ShouldNotBeNil)
				mocks.DynamoDB.AssertExpectations(t)
			})
		})

		Convey("When the call to dynamo fails", func() {
			Convey("It should return an error", func() {
				key := "hide_follows"
				val := true
				mocks.DynamoDB.On("SetAlertPreferences", ctx, channelID, key, val).Once().Return(nil, errors.New("dynamo error"))

				_, err := manager.SetAlertPrefs(ctx, channelID, key, val)
				So(err, ShouldNotBeNil)
				mocks.DynamoDB.AssertExpectations(t)
			})
		})

		Convey("When the call to dynamo succeeds", func() {
			Convey("It should return the new prefs", func() {
				key := "hide_follows"
				val := true
				resp := &dynamodb.AlertPreferences{
					ChannelID:      channelID,
					DNDModeEnabled: true,
					HideFollows:    true,
					HideRaids:      false,
				}
				mocks.DynamoDB.On("SetAlertPreferences", ctx, channelID, key, val).Once().Return(resp, nil)

				alertPrefs, err := manager.SetAlertPrefs(ctx, channelID, key, val)
				So(err, ShouldBeNil)
				So(alertPrefs, ShouldResemble, resp)
				mocks.DynamoDB.AssertExpectations(t)
			})
		})
	})
}

func TestUpdateAndPublishAlert(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := managerMocks()
	manager := mockManager(mocks)
	ctx := context.Background()

	mocks.Statsd.On("ExecutionTime", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When calling UpdateAndPublishAlert", t, func() {
		Convey("When the channel id is empty", func() {
			Convey("The call should fail", func() {
				invalidChannelID := ""

				mocks.DynamoDB.AssertNotCalled(t, "SetAlertStatus", ctx, invalidChannelID, activityID, newStatus)
				mocks.Pubsub.AssertNotCalled(t, "PublishAlert", ctx, invalidChannelID, mock.Anything)
				mocks.Users.AssertNotCalled(t, "GetUser", ctx, mock.Anything)
				result, err := manager.UpdateAndPublishAlert(ctx, invalidChannelID, activityID, newStatus)

				So(err, ShouldNotBeNil)
				So(result, ShouldBeNil)
				mocks.DynamoDB.AssertExpectations(t)
				mocks.Pubsub.AssertExpectations(t)
				mocks.Users.AssertExpectations(t)
			})
		})

		Convey("When the activity id is empty", func() {
			Convey("The call should fail", func() {
				invalidActivityID := ""

				mocks.DynamoDB.AssertNotCalled(t, "SetAlertStatus", ctx, channelID, invalidActivityID, newStatus)
				mocks.Pubsub.AssertNotCalled(t, "PublishAlert", ctx, channelID, mock.Anything)
				mocks.Users.AssertNotCalled(t, "GetUser", ctx, mock.Anything)
				result, err := manager.UpdateAndPublishAlert(ctx, channelID, invalidActivityID, newStatus)

				So(err, ShouldNotBeNil)
				So(result, ShouldBeNil)
				mocks.DynamoDB.AssertExpectations(t)
				mocks.Pubsub.AssertExpectations(t)
				mocks.Users.AssertExpectations(t)
			})
		})

		Convey("When the new status is invalid", func() {
			Convey("The call should fail", func() {
				invalidStatus := "invalid-status"

				mocks.DynamoDB.AssertNotCalled(t, "SetAlertStatus", ctx, channelID, activityID, invalidStatus)
				mocks.Pubsub.AssertNotCalled(t, "PublishAlert", ctx, channelID, mock.Anything)
				mocks.Users.AssertNotCalled(t, "GetUser", ctx, mock.Anything)
				result, err := manager.UpdateAndPublishAlert(ctx, channelID, activityID, invalidStatus)

				So(err, ShouldNotBeNil)
				So(result, ShouldBeNil)
				mocks.DynamoDB.AssertExpectations(t)
				mocks.Pubsub.AssertExpectations(t)
				mocks.Users.AssertExpectations(t)
			})
		})

		Convey("When the input is valid, but dynamo fails", func() {
			Convey("The call should fail", func() {
				mocks.DynamoDB.On("SetAlertStatus", ctx, channelID, activityID, newStatus).Once().Return(nil, errors.New("dynamo error"))
				mocks.Pubsub.AssertNotCalled(t, "PublishAlert", ctx, channelID, mock.Anything)
				mocks.Users.AssertNotCalled(t, "GetUser", ctx, mock.Anything)
				result, err := manager.UpdateAndPublishAlert(ctx, channelID, activityID, newStatus)

				So(err, ShouldNotBeNil)
				So(result, ShouldBeNil)
				mocks.DynamoDB.AssertExpectations(t)
				mocks.Pubsub.AssertExpectations(t)
				mocks.Users.AssertExpectations(t)
			})
		})

		Convey("When the input is valid, and dynamo succeeds, but pubsub fails", func() {
			Convey("The call should fail", func() {
				followerID := "000000"
				dynamoResp := &dynamodb.Activity{
					ID:         activityID,
					Type:       activity.TypeFollow,
					Timestamp:  time.Now(),
					FollowerID: &followerID,
				}
				usersResp := types.User{
					ID:          followerID,
					Login:       "test-login",
					DisplayName: "displayname1",
				}

				mocks.Users.On("GetUser", ctx, followerID).Once().Return(usersResp, nil)
				mocks.DynamoDB.On("SetAlertStatus", ctx, channelID, activityID, newStatus).Once().Return(dynamoResp, nil)
				mocks.Pubsub.On("PublishAlert", ctx, channelID, mock.Anything).Once().Return(errors.New("pubsub error"))
				result, err := manager.UpdateAndPublishAlert(ctx, channelID, activityID, newStatus)

				So(err, ShouldNotBeNil)
				So(result, ShouldBeNil)
				mocks.DynamoDB.AssertExpectations(t)
				mocks.Pubsub.AssertExpectations(t)
				mocks.Users.AssertExpectations(t)
			})
		})

		Convey("When the input is valid, dynamo succeeds and but converting to pubsub fails", func() {
			Convey("The call should return an error", func() {
				channelID := "333333"
				followerID := "999999"
				dynamoResp := &dynamodb.Activity{
					ID:         activityID,
					Type:       activity.TypeFollow,
					Timestamp:  time.Now(),
					FollowerID: &followerID,
				}

				mocks.Users.On("GetUser", ctx, followerID).Once().Return(types.User{}, errors.New("users error"))
				mocks.DynamoDB.On("SetAlertStatus", ctx, channelID, activityID, newStatus).Once().Return(dynamoResp, nil)
				mocks.Pubsub.AssertNotCalled(t, "PublishAlert", ctx, channelID, mock.Anything)
				result, err := manager.UpdateAndPublishAlert(ctx, channelID, activityID, newStatus)

				So(err, ShouldNotBeNil)
				So(result, ShouldBeNil)
				mocks.DynamoDB.AssertExpectations(t)
				mocks.Pubsub.AssertExpectations(t)
				mocks.Users.AssertExpectations(t)
			})
		})

		Convey("When the input is valid, dynamo succeeds and pubsub succeeds", func() {
			Convey("The call should return the activity and no error", func() {
				followerID := "999999"
				dynamoResp := &dynamodb.Activity{
					ID:         activityID,
					Type:       activity.TypeFollow,
					Timestamp:  time.Now(),
					FollowerID: &followerID,
				}
				usersResp := types.User{
					ID:          followerID,
					Login:       "test-login",
					DisplayName: "displayname1",
				}

				mocks.Users.On("GetUser", ctx, followerID).Once().Return(usersResp, nil)
				mocks.DynamoDB.On("SetAlertStatus", ctx, channelID, activityID, newStatus).Once().Return(dynamoResp, nil)
				mocks.Pubsub.On("PublishAlert", ctx, channelID, mock.Anything).Once().Return(nil)
				result, err := manager.UpdateAndPublishAlert(ctx, channelID, activityID, newStatus)

				So(err, ShouldBeNil)
				So(result, ShouldNotBeNil)
				So(result, ShouldResemble, dynamoResp)
				mocks.DynamoDB.AssertExpectations(t)
				mocks.Pubsub.AssertExpectations(t)
				mocks.Users.AssertExpectations(t)
			})
		})
	})
}
