package communitypoints

import (
	"context"
	"net/http"
	"time"

	"code.justin.tv/cb/sauron/internal/clients"
	copotwirp "code.justin.tv/chat/copo/proto/copo"

	"github.com/pkg/errors"
)

// CopoService is the interface for acessing copo data
type CopoService interface {
	GetChannelPointsName(ctx context.Context, channelID string) (string, error)
}

// Client wraps the CopoService service client.
type Client struct {
	Copo copotwirp.Copo
}

// NewClient creates a new client wrapping the users service internal client
func NewClient(host string) *Client {
	return &Client{
		Copo: copotwirp.NewCopoProtobufClient(host, &http.Client{
			Timeout: 1 * time.Second,
		}),
	}
}

func (c *Client) GetChannelPointsName(ctx context.Context, channelID string) (string, error) {
	var resp *copotwirp.GetChannelSettingsResponse

	err := clients.Retry(ctx, clients.NewBackoffWithJitterRetryPolicy(), func() (bool, error) {
		var getError error
		resp, getError = c.Copo.GetChannelSettings(ctx, &copotwirp.GetChannelSettingsRequest{
			ChannelId: channelID,
		})
		return false, getError
	})

	if err != nil {
		return "", errors.Wrap(err, "copo: failed to get channel settings")
	}

	settings := resp.GetSettings()

	if settings == nil {
		return "", errors.New("copo: channel settings are empty")
	}

	return settings.GetName(), nil
}
