package dynamodb

import (
	"context"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/pkg/errors"
)

const timeFormat = "2006-01-02T15:04:05.000Z"

// GetAlertPreferences retrieves the alert preferences for a user
func (c *Client) GetAlertPreferences(ctx context.Context, channelID string) (*AlertPreferences, error) {
	input := &dynamodb.GetItemInput{
		Key: map[string]*dynamodb.AttributeValue{
			"channel_id": {
				S: aws.String(channelID),
			},
		},
		TableName: aws.String(c.alertPrefsTable),
	}

	var prefs *AlertPreferences

	output, err := c.dynamoDB.GetItemWithContext(ctx, input)
	if err != nil {
		return nil, errors.Wrap(err, "dynamodb: failed to get alert preferences")
	}

	// If there are no settings present, we return a struct with the channelID
	// and the default values for each setting, assuming the user hasn't set anything yet.
	if len(output.Item) == 0 {
		return &AlertPreferences{
			ChannelID: channelID,
		}, nil
	}

	if err := dynamodbattribute.UnmarshalMap(output.Item, &prefs); err != nil {
		return nil, errors.Wrap(err, "dynamodb: failed to convert to alert preferences")
	}

	return prefs, nil
}

// SetAlertPreferences sets the alert preferences for a user
func (c *Client) SetAlertPreferences(ctx context.Context, channelID string, prefName string, prefVal bool) (*AlertPreferences, error) {
	input := &dynamodb.UpdateItemInput{
		TableName: aws.String(c.alertPrefsTable),
		Key: map[string]*dynamodb.AttributeValue{
			"channel_id": {S: aws.String(channelID)},
		},
		ExpressionAttributeNames: map[string]*string{
			"#P": aws.String(prefName),
			"#T": aws.String("last_modified"),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":p": {BOOL: aws.Bool(prefVal)},
			":t": {S: aws.String(time.Now().Format(timeFormat))},
		},
		UpdateExpression: aws.String("SET #P = :p, #T = :t"),
		ReturnValues:     aws.String("ALL_NEW"),
	}

	var prefs *AlertPreferences

	output, err := c.dynamoDB.UpdateItemWithContext(ctx, input)

	if err != nil {
		return nil, errors.Wrap(err, "dynamodb: failed to update preference")
	}

	if err := dynamodbattribute.UnmarshalMap(output.Attributes, &prefs); err != nil {
		return nil, errors.Wrap(err, "dynamodb: failed to convert to alert preferences")
	}

	return prefs, nil
}

func (c *Client) DeleteAlertPreferences(ctx context.Context, channelID string) error {
	input := &dynamodb.DeleteItemInput{
		TableName: aws.String(c.alertPrefsTable),
		Key: map[string]*dynamodb.AttributeValue{
			"channel_id": {S: aws.String(channelID)},
		},
	}

	_, err := c.dynamoDB.DeleteItemWithContext(ctx, input)
	if err != nil {
		return errors.Wrap(err, "dynamodb: failed to delete preference")
	}

	return nil
}
