package dynamodb

import (
	"context"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/pkg/errors"
)

const (
	day        = 24 * time.Hour
	expiration = 28 * day

	maxBatchWriteItems = 25 // https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_BatchWriteItem.html
)

// GetActivities queries a given channel's activities
// up to the limit or until there are no more records (i.e. null LastEvaluatedKey).
func (c *Client) GetActivities(ctx context.Context, channelID string, before time.Time, limit int) ([]Activity, error) {
	input := &dynamodb.QueryInput{
		KeyConditionExpression: aws.String("channel_id = :channelID AND #T < :before"),
		ExpressionAttributeNames: map[string]*string{
			"#T": aws.String("timestamp"),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":channelID": {
				S: aws.String(channelID),
			},
			":before": {
				S: aws.String(before.Format(time.RFC3339Nano)),
			},
		},
		ExclusiveStartKey: nil,
		Limit:             aws.Int64(int64(limit)),
		ScanIndexForward:  aws.Bool(false),
		TableName:         aws.String(c.activityTable),
	}

	var activities []Activity

	for {
		output, err := c.dynamoDB.QueryWithContext(ctx, input)
		if err != nil {
			return nil, errors.Wrap(err, "dynamodb: failed to get activities")
		}

		for _, item := range output.Items {
			var activity Activity

			if err := dynamodbattribute.UnmarshalMap(item, &activity); err != nil {
				return nil, errors.Wrap(err, "dynamodb: failed to convert to activity")
			}

			activities = append(activities, activity)

			if len(activities) == limit {
				return activities, nil
			}
		}

		if output.LastEvaluatedKey == nil {
			break
		}

		input.SetExclusiveStartKey(output.LastEvaluatedKey)
	}

	return activities, nil
}

// DeleteActivities hard deletes a given channel's activities from the dynamoDB table.
func (c *Client) DeleteActivities(ctx context.Context, channelID string) error {
	queryInput := &dynamodb.QueryInput{
		KeyConditionExpression: aws.String("channel_id = :channelID"),
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":channelID": {
				S: aws.String(channelID),
			},
		},
		ExclusiveStartKey: nil,
		ExpressionAttributeNames: map[string]*string{
			"#T": aws.String("timestamp"),
		},
		ProjectionExpression: aws.String("#T"),
		TableName:            aws.String(c.activityTable),
	}

	var timestamps []string

	for {
		output, err := c.dynamoDB.QueryWithContext(ctx, queryInput)
		if err != nil {
			return errors.Wrap(err, "dynamodb: failed to get activities to delete")
		}

		for _, item := range output.Items {
			var timestamp string

			if err := dynamodbattribute.Unmarshal(item["timestamp"], &timestamp); err != nil {
				return errors.Wrap(err, "dynamodb: failed to convert to timestamp to delete")
			}

			timestamps = append(timestamps, timestamp)
		}

		if output.LastEvaluatedKey == nil {
			break
		}

		queryInput.SetExclusiveStartKey(output.LastEvaluatedKey)
	}

	if len(timestamps) == 0 {
		return nil
	}

	writeRequests := make([]*dynamodb.WriteRequest, 0, maxBatchWriteItems)

	for idx, timestamp := range timestamps {
		deleteRequest := &dynamodb.DeleteRequest{
			Key: map[string]*dynamodb.AttributeValue{
				"channel_id": {S: aws.String(channelID)},
				"timestamp":  {S: aws.String(timestamp)},
			},
		}
		writeRequests = append(writeRequests, &dynamodb.WriteRequest{
			DeleteRequest: deleteRequest,
		})

		if len(writeRequests) == maxBatchWriteItems || idx == len(timestamps)-1 {
			for {
				output, err := c.dynamoDB.BatchWriteItemWithContext(ctx, &dynamodb.BatchWriteItemInput{
					RequestItems: map[string][]*dynamodb.WriteRequest{
						c.activityTable: writeRequests,
					},
				})
				if err != nil {
					return errors.Wrap(err, "dynamodb: failed to delete activities")
				}

				writeRequests = output.UnprocessedItems[c.activityTable]
				if len(writeRequests) == 0 {
					writeRequests = make([]*dynamodb.WriteRequest, 0, maxBatchWriteItems)
					break
				}
			}
		}
	}

	return nil
}

func (c *Client) put(ctx context.Context, activity Activity) error {
	item, err := dynamodbattribute.MarshalMap(activity)
	if err != nil {
		return errors.Wrap(err, "dynamodb: failed to marshal activity")
	}

	_, err = c.dynamoDB.PutItemWithContext(ctx, &dynamodb.PutItemInput{
		Item:      item,
		TableName: aws.String(c.activityTable),
	})
	if err != nil {
		return errors.Wrap(err, "dynamodb: failed to put item")
	}

	return nil
}
