package dynamodb

import (
	"context"
	"fmt"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/awserr"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbattribute"
	"github.com/aws/aws-sdk-go/service/dynamodb/expression"
	"github.com/pkg/errors"

	"code.justin.tv/cb/sauron/activity"
	log "github.com/sirupsen/logrus"
)

const (
	rateLimitPeriod     = 24 * time.Hour
	rateLimitExpiration = rateLimitPeriod
)

// An allowlist for test accounts to bypass rate limits. This allows people
// to test features like alerts without dealing with rate limits.
// This bypass applies to the actual acount being followed, not the follower itself.
var rateLimitBypassAllowlist = map[string]bool {
	"237755203": true, // packofjackals
	"30572359": true, // ondaryn
	"48276979": true, // flavioliravioli
	"168032851": true, // null_n
	"263322421": true, // chippytest
	"610408892": true, // viohime
	"456315850": true, // moluisthebeststreamer
	"72182602": true, // qa_partner2
	"618864939": true, // treeshuhtests
}

// ShouldRateLimitFollow inserts a follow activity into the rate limit table,
// and determines whether activity should be published in activity feed and as an alert
func (c *Client) ShouldInsertAndPublishFollow(ctx context.Context, channelID, sourceID string, timestamp time.Time) (bool, error) {
	if rateLimitBypassAllowlist[channelID] {
		return true, nil
	}

	err := c.putRateLimit(ctx, timestamp, RateLimit{
		Key:         c.createFollowKey(channelID, sourceID),
		Timestamp:   c.formatTimestamp(timestamp),
		ExpiresUnix: timestamp.Add(rateLimitExpiration).Unix(),

		ChannelID: channelID,
		ActorID:   sourceID,
		Type:      activity.TypeFollow,
	})

	if err != nil {
		// Failed to insert because follow event was rate limited
		if aerr, ok := err.(awserr.Error); ok && aerr.Code() == dynamodb.ErrCodeConditionalCheckFailedException {
			return false, nil
		}

		// Unexpected error not from being rate limited
		return false, errors.Wrap(err, "dynamodb: failed to put rate limit item")
	}

	return true, nil
}

// DeleteRateLimitFollow deletes a follow activity from the rate limit table.
func (c *Client) DeleteRateLimitFollow(ctx context.Context, channelID, sourceID string) {
	if rateLimitBypassAllowlist[channelID] {
		return
	}

	primaryKey := c.createFollowKey(channelID, sourceID)

	err := c.deleteRateLimit(ctx, primaryKey)
	if err != nil {
		logger := log.WithFields(log.Fields{
			"dynamodb": "rate_limit",
			"key":      primaryKey,
		})
		logger.WithError(err).Error("dynamodb: failed to delete rate limit item")
		c.statsd.GoIncrement("event.follow.ratelimit.delete.error", 1)
	}

	c.statsd.GoIncrement("event.follow.ratelimit.delete.success", 1)
}

func (c *Client) putRateLimit(ctx context.Context, entryTimestamp time.Time, rateLimitEntry RateLimit) error {
	item, err := dynamodbattribute.MarshalMap(rateLimitEntry)
	if err != nil {
		return errors.Wrap(err, "dynamodb: failed to marshal activity")
	}

	// Creates conditional expression for when PutItem would succeed
	minTimestamp := c.formatTimestamp(entryTimestamp.Add(-1 * rateLimitPeriod))
	cond := expression.Name("timestamp").AttributeNotExists().Or(expression.Name("timestamp").LessThanEqual(expression.Value(minTimestamp)))
	expr, err := expression.NewBuilder().WithCondition(cond).Build()
	if err != nil {
		return errors.Wrap(err, "dynamodb: failed build PutItem expression")
	}

	_, err = c.dynamoDB.PutItemWithContext(ctx, &dynamodb.PutItemInput{
		TableName:                 aws.String(c.rateLimitTable),
		ConditionExpression:       expr.Condition(),
		ExpressionAttributeNames:  expr.Names(),
		ExpressionAttributeValues: expr.Values(),
		Item:                      item,
	})

	return err
}

func (c *Client) deleteRateLimit(ctx context.Context, primaryKey string) error {
	_, err := c.dynamoDB.DeleteItemWithContext(ctx, &dynamodb.DeleteItemInput{
		TableName: aws.String(c.rateLimitTable),
		Key: map[string]*dynamodb.AttributeValue{
			"key": {
				S: aws.String(primaryKey),
			},
		},
	})
	return err
}

// Create the key for the rate limit table that is used to identify each follow event
func (c *Client) createFollowKey(channelID, sourceID string) string {
	return fmt.Sprintf("follow:%s:%s", channelID, sourceID)
}

// Formats timestamp into RFC3339 string format to allow use of comparison operators
func (c *Client) formatTimestamp(timestamp time.Time) string {
	return timestamp.UTC().Format(time.RFC3339)
}
