package liveline

import (
	"context"

	"code.justin.tv/cb/sauron/internal/clients/stats"
	"code.justin.tv/discovery/liveline/proto/liveline"
	"code.justin.tv/foundation/twitchclient"
	"github.com/afex/hystrix-go/hystrix"

	log "github.com/sirupsen/logrus"
)

const getLiveStreams = "liveline.get_live_streams"

// Liveline provides methods for checking channel liveness and stream information
type Liveline interface {
	IsLive(ctx context.Context, channelID string) (bool, error)
}

// Client wraps the liveline client
type Client struct {
	statsd   stats.StatSender
	liveline liveline.Liveline
}

func init() {
	hystrix.Configure(map[string]hystrix.CommandConfig{
		getLiveStreams: {
			Timeout:                1000,
			MaxConcurrentRequests:  50,
			RequestVolumeThreshold: 100,
			SleepWindow:            5000,
			ErrorPercentThreshold:  25,
		},
	})
}

// NewClient creates a new client wrapper
func NewClient(livelineHost string, statsClient stats.StatSender) *Client {
	httpClient := twitchclient.NewHTTPClient(
		twitchclient.ClientConf{},
	)

	return &Client{
		statsd:   statsClient,
		liveline: liveline.NewLivelineProtobufClient(livelineHost, httpClient),
	}
}

// IsLive returns true if a channel is live
func (c *Client) IsLive(ctx context.Context, channelID string) (bool, error) {
	var liveStreams *liveline.StreamsResponse
	err := hystrix.DoC(ctx, getLiveStreams, func(ctx context.Context) error {
		resp, err := c.liveline.GetStreamsByChannelIDs(ctx, &liveline.StreamsByChannelIDsRequest{
			ChannelIds: []string{channelID},
			SortKey:    liveline.SortField_SORTFIELD_UNSET,
			Order:      liveline.SortOrder_SORTORDER_UNSET,
		})

		if err != nil {
			liveStreams = &liveline.StreamsResponse{}
			c.statsd.GoIncrement("liveline.streams_by_channel_ids.error", 1)
			return err
		}
		c.statsd.GoIncrement("liveline.streams_by_channel_ids.success", 1)
		liveStreams = resp
		return nil
	}, func(ctx context.Context, herr error) error {
		// error fallback
		log.WithError(herr).Error("hystrix circuit open: error getting live streams from liveline")
		liveStreams = &liveline.StreamsResponse{}
		return nil
	})

	if err != nil {
		return false, err
	}

	return len(liveStreams.Streams) > 0, nil
}
