package pdms

import (
	"context"
	"net/http"
	"time"

	"github.com/aws/aws-sdk-go/aws/endpoints"

	"code.justin.tv/amzn/PDMSLambdaTwirp"
	twirplambdatransport "code.justin.tv/amzn/TwirpGoLangAWSTransports/lambda"
	"github.com/pkg/errors"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/lambda"
)

// Service ID from Catalog https://catalog.xarth.tv/services/532/details
// This is the ID that PDMS uses when whitelisting services.
const serviceID string = "532"

type Client struct {
	client PDMSLambdaTwirp.PDMSService
}

type ClientConfig struct {
	CallerRoleArn string
	LambdaArn     string
}

func NewClient(config ClientConfig) *Client {
	sess := session.Must(session.NewSessionWithOptions(session.Options{
		Config: aws.Config{
			Region:              aws.String("us-west-2"),
			STSRegionalEndpoint: endpoints.RegionalSTSEndpoint,
			HTTPClient:          &http.Client{Timeout: 10 * time.Second},
		},
	}))

	creds := stscreds.NewCredentials(sess, config.CallerRoleArn)
	pdmsLambdaCli := lambda.New(sess, &aws.Config{Credentials: creds})
	pdmsTransport := twirplambdatransport.NewClient(pdmsLambdaCli, config.LambdaArn)
	client := PDMSLambdaTwirp.NewPDMSServiceProtobufClient("", pdmsTransport)

	return &Client{client: client}
}

func (c *Client) ReportDeletion(context context.Context, userID string) error {

	request := PDMSLambdaTwirp.ReportDeletionRequest{
		UserId:    userID,
		ServiceId: serviceID,
	}

	if _, err := c.client.ReportDeletion(context, &request); err != nil {
		return errors.Wrap(err, "pdms: failed to call report deletion")
	}

	return nil
}
