package pubsub

import (
	"context"
	"encoding/json"
	"fmt"

	"code.justin.tv/cb/sauron/activity/pubsub"
	"code.justin.tv/cb/sauron/internal/clients"
	"code.justin.tv/chat/pubsub-go-pubclient/client"
	"code.justin.tv/foundation/twitchclient"
	"github.com/pkg/errors"
)

const (
	alertTopicName    = "dashboard-alert-status"
	activityTopicName = "dashboard-activity-feed"
	productionEnv     = "production"
)

// Client wraps the Pubsub client.
type Client struct {
	pubsub              client.PubClient
	activityTopicPrefix string
	alertTopicPrefix    string
}

// NewClient instantiates a Client.
func NewClient(env, host string) (*Client, error) {
	pubsubClient, err := client.NewPubClient(twitchclient.ClientConf{
		Host: host,
	})
	if err != nil {
		return nil, errors.Wrap(err, "failed to initialize pubsub client")
	}

	// To determine the topic names, we check the env and prepend "pubsubtest"
	// if we're not running in prod. This allows us to listen to the topics
	// locally for testing.
	activityTopic := activityTopicName
	alertTopic := alertTopicName
	if env != productionEnv {
		activityTopic = fmt.Sprintf("pubsubtest.%s", activityTopicName)
		alertTopic = fmt.Sprintf("pubsubtest.%s", alertTopicName)
	}

	return &Client{
		pubsub:              pubsubClient,
		activityTopicPrefix: activityTopic,
		alertTopicPrefix:    alertTopic,
	}, nil
}

func (c *Client) PublishAlert(ctx context.Context, channelID string, alert Alert) error {
	alert.Type = alertTopicName
	bytes, err := json.Marshal(alert)
	if err != nil {
		return errors.Wrap(err, "pubsub: failed to json marshal alert")
	}

	topics := []string{getTopicName(c.alertTopicPrefix, channelID)}
	message := string(bytes)

	err = clients.Retry(ctx, clients.NewBackoffWithJitterRetryPolicy(), func() (bool, error) {
		return false, c.pubsub.Publish(ctx, topics, message, nil)
	})

	if err != nil {
		return errors.Wrap(err, "pubsub: failed to publish alert")
	}

	return nil
}

func (c *Client) publish(ctx context.Context, channelID string, activity pubsub.Activity) error {
	bytes, err := json.Marshal(activity)
	if err != nil {
		return errors.Wrap(err, "pubsub: failed to json marshal activity")
	}

	topics := []string{getTopicName(c.activityTopicPrefix, channelID)}
	message := string(bytes)

	err = clients.Retry(ctx, clients.NewBackoffWithJitterRetryPolicy(), func() (bool, error) {
		return false, c.pubsub.Publish(ctx, topics, message, nil)
	})

	if err != nil {
		return errors.Wrap(err, "pubsub: failed to publish activity")
	}

	return nil
}

func getTopicName(prefix, channelID string) string {
	return fmt.Sprintf("%s.%s", prefix, channelID)
}
