package clients

import (
	"context"
	"errors"
	"testing"
	"time"

	. "github.com/smartystreets/goconvey/convey"
)

type mockRetryPolicy struct {
	retryCount int
}

func newMockRetryPolicy(retryCount int) RetryPolicy {
	return &mockRetryPolicy{
		retryCount: retryCount,
	}
}

func (p *mockRetryPolicy) GetRetryCount() int {
	return p.retryCount
}

func (p *mockRetryPolicy) GetSleepDuration() time.Duration {
	return 0 * time.Millisecond
}

func TestRetry(t *testing.T) {
	ctx := context.Background()

	Convey("Retry", t, func() {
		Convey("succeeds when no error is returned", func() {
			actualRetryCount := 0
			retryPolicy := newMockRetryPolicy(3)

			err := Retry(ctx, retryPolicy, func() (bool, error) {
				actualRetryCount++
				return false, nil
			})

			So(actualRetryCount, ShouldEqual, 1)
			So(err, ShouldBeNil)
		})

		Convey("retries and then fails when error is returned", func() {
			actualRetryCount := 0
			retryPolicy := newMockRetryPolicy(3)

			err := Retry(ctx, retryPolicy, func() (bool, error) {
				actualRetryCount++
				return false, errors.New("Test error")
			})

			So(actualRetryCount, ShouldEqual, 4)
			So(err, ShouldBeError)
		})

		Convey("does not retry and then fails when an expected error is returned", func() {
			actualRetryCount := 0
			retryPolicy := newMockRetryPolicy(3)

			err := Retry(ctx, retryPolicy, func() (bool, error) {
				actualRetryCount++
				return true, errors.New("Test error")
			})

			So(actualRetryCount, ShouldEqual, 1)
			So(err, ShouldBeError)
		})
	})
}

func TestBackoffRetryPolicy(t *testing.T) {
	Convey("Retry policy ", t, func() {
		Convey("backoff increases with each call to GetSleepDuration", func() {
			retryPolicy := NewBackoffWithJitterRetryPolicy()

			sleepDuration := retryPolicy.GetSleepDuration()
			So(sleepDuration, ShouldBeBetweenOrEqual, 100*time.Millisecond, 200*time.Millisecond)

			sleepDuration = retryPolicy.GetSleepDuration()
			So(sleepDuration, ShouldBeBetweenOrEqual, 200*time.Millisecond, 400*time.Millisecond)

			sleepDuration = retryPolicy.GetSleepDuration()
			So(sleepDuration, ShouldBeBetweenOrEqual, 400*time.Millisecond, 800*time.Millisecond)
		})
	})
}
