package secrets

import (
	"encoding/base64"
	"encoding/json"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/secretsmanager"
	"github.com/pkg/errors"
)

// Secrets is the interface for getting secret values.
type Secrets interface {
	GetSecret(secretName string) (string, error)
}

// Manager wraps the secretsmanager client
type Manager struct {
	manager *secretsmanager.SecretsManager
}

// NewManager creates a manager for retrieving secrets from AWS secret manager
func NewManager() (*Manager, error) {
	awsConfig := aws.NewConfig().WithRegion("us-west-2")
	sess, err := session.NewSession(awsConfig)
	if err != nil {
		return nil, errors.Wrap(err, "secrets: failed to initialize new aws session")
	}

	manager := secretsmanager.New(sess)

	return &Manager{
		manager: manager,
	}, nil
}

// GetSecret retrieves a secret for a given name, and a given key. The key is used
// because Secret Manager stores each secret under a name as a key/value pair. Providing
// the key here allows this method to return only the plaintext value, so the caller
// doesn't have to handle it
func (c *Manager) GetSecret(secretName string, secretKey string) (string, error) {
	input := &secretsmanager.GetSecretValueInput{
		SecretId: aws.String(secretName),
	}

	result, err := c.manager.GetSecretValue(input)
	if err != nil {
		return "", errors.Wrap(err, "secrets: failed to retrieve secret value")
	}

	plaintext, err := decodeSecret(result, secretKey)
	if err != nil {
		return "", errors.Wrap(err, "secrets: failed to decode secret value")
	}

	return plaintext, nil
}

// decodeSecret returns the plaintext value of a secret that has been retrieved. For secrets
// that are strings, no work needs to be done here. If a secret is binary, we have to decode
// it from base64 and return the plaintext
func decodeSecret(result *secretsmanager.GetSecretValueOutput, secretKey string) (string, error) {
	if result.SecretString != nil {
		jsonResult := *result.SecretString
		var secretPair map[string]string

		err := json.Unmarshal([]byte(jsonResult), &secretPair)
		if err != nil {
			return "", err
		}

		return secretPair[secretKey], nil
	}

	var decodedBinarySecret string
	decodedBinarySecretBytes := make([]byte, base64.StdEncoding.DecodedLen(len(result.SecretBinary)))
	len, err := base64.StdEncoding.Decode(decodedBinarySecretBytes, result.SecretBinary)
	if err != nil {
		return "", err
	}

	decodedBinarySecret = string(decodedBinarySecretBytes[:len])

	return decodedBinarySecret, nil
}
