package stats

import (
	"context"
	"fmt"
	"sync"
	"time"

	"code.justin.tv/common/gometrics"
	"github.com/afex/hystrix-go/plugins"
	"github.com/cactus/go-statsd-client/statsd"
	"github.com/pkg/errors"

	metricCollector "github.com/afex/hystrix-go/hystrix/metric_collector"
	log "github.com/sirupsen/logrus"
)

// StatSender type alias's the exported client interface from Statsd service.
type StatSender interface {
	statsd.Statter
	ExecutionTime(statName string, duration time.Duration)
	GoExecutionTime(statName string, duration time.Duration)
	Increment(statName string, value int64)
	GoIncrement(statName string, value int64)
	SendGauge(statName string, value int64)
	GoOther(callback func())
	Shutdown(ctx context.Context) error
}

// Client is the statsd client
type Client struct {
	statsd.Statter
	wg sync.WaitGroup
}

const (
	flushInterval        = 1 * time.Second
	gometricsMonitorRate = 1 * time.Second
)

// NewClient instantiates a new statsd client
func NewClient(host, env string) (*Client, error) {
	prefix := fmt.Sprintf("cb.sauron.%v", env)

	statter, err := statsd.NewBufferedClient(host, prefix, flushInterval, 512)
	if err != nil {
		return nil, errors.Wrap(err, "statsd: failed to instantiate buffered client")
	}

	log.Info(fmt.Sprintf("Connected to StatsD at %s with prefix %s", host, prefix))

	gometrics.Monitor(statter, gometricsMonitorRate)

	err = initHystrix(host, prefix)
	if err != nil {
		return nil, errors.Wrap(err, "statsd: failed to instantiate hystrix statsd collector")
	}

	return &Client{
		Statter: statter,
	}, nil
}

// ExecutionTime records the execution time given a duration.
func (c *Client) ExecutionTime(statName string, duration time.Duration) {
	err := c.TimingDuration(statName, duration, 1.0)
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"stat_name": statName,
			"duration":  duration,
		}).Warn("statsd: failed to send timing duration")
	}
}

// ExecutionTime records the execution time given a duration.
func (c *Client) GoExecutionTime(statName string, duration time.Duration) {
	c.wg.Add(1)
	go func() {
		defer c.wg.Done()
		c.ExecutionTime(statName, duration)
	}()
}

// Increment increments the stat by the given value.
func (c *Client) Increment(statName string, value int64) {
	err := c.Inc(statName, value, 1.0)
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"stat_name": statName,
			"value":     value,
		}).Warn("statsd: failed to send increment")
	}
}

// Increment increments the stat by the given value.
func (c *Client) GoIncrement(statName string, value int64) {
	c.wg.Add(1)
	go func() {
		defer c.wg.Done()
		c.Increment(statName, value)
	}()
}

// SendGauge sends a gauge value to statsd. This is wrapped in the client for brevity,
// because error handling is required but takes up lots of space for the caller.
func (c *Client) SendGauge(statName string, value int64) {
	err := c.Gauge(statName, value, 1.0)
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"stat_name": statName,
			"value":     value,
		}).Warn("statsd: failed to send gauge")
	}
}

func (c *Client) GoOther(callback func()) {
	c.wg.Add(1)
	go func() {
		defer c.wg.Done()
		callback()
	}()
}

func (c *Client) Shutdown(ctx context.Context) error {
	ch := make(chan struct{})
	go func() {
		c.wg.Wait()
		close(ch)
	}()

	select {
	case <-ch:
		return nil
	case <-ctx.Done():
		return errors.New("timeout")
	}
}

func initHystrix(host, prefix string) error {
	c, err := plugins.InitializeStatsdCollector(&plugins.StatsdCollectorConfig{
		StatsdAddr: host,
		Prefix:     prefix + ".hystrix",
	})
	if err != nil {
		return err
	}

	metricCollector.Registry.Register(c.NewStatsdCollector)
	return nil
}
