package subscriptions

import (
	"context"
	"net/http"
	"time"

	"code.justin.tv/cb/sauron/internal/clients"
	substwirp "code.justin.tv/revenue/subscriptions/twirp"
	"github.com/pkg/errors"
)

// Expected errors:
var (
	ErrProductNotFound = errors.New("subscriptions: product not found")
)

// SubService is the interface for acessing subscriptions data
type SubService interface {
	GetProductTier(ctx context.Context, productID string) (string, error)
}

// Client wraps the SubService service client.
type Client struct {
	Subscriptions substwirp.Subscriptions
}

// NewClient creates a new client wrapping the users service internal client
func NewClient(host string) *Client {
	return &Client{
		Subscriptions: substwirp.NewSubscriptionsProtobufClient(host, &http.Client{
			Timeout: 1 * time.Second,
		}),
	}
}

// GetProductTier queries for a subscription product by ID and returns the product's tier.
func (c *Client) GetProductTier(ctx context.Context, productID string) (string, error) {
	req := &substwirp.GetProductsByIDsRequest{
		ProductIds: []string{productID},
	}

	var resp *substwirp.GetProductsByIDsResponse

	err := clients.Retry(ctx, clients.NewBackoffWithJitterRetryPolicy(), func() (bool, error) {
		var getError error
		resp, getError = c.Subscriptions.GetProductsByIDs(ctx, req)
		return false, getError
	})

	if err != nil {
		return "", errors.Wrap(err, "subscriptions: failed to get products by ids")
	}

	for _, product := range resp.GetProducts() {
		if product.GetId() == productID {
			return product.GetTier(), nil
		}
	}

	return "", ErrProductNotFound
}
