package users

import (
	"context"
	"time"

	"code.justin.tv/cb/sauron/types"

	"code.justin.tv/cb/sauron/internal/clients"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/web/users-service/client"
	"code.justin.tv/web/users-service/client/usersclient_internal"
	"code.justin.tv/web/users-service/models"
	"github.com/pkg/errors"
)

// Users is the interface for users service information.
type Users interface {
	GetUser(ctx context.Context, id string) (types.User, error)
}

// Client wraps the user service internal client.
type Client struct {
	usersclient_internal.InternalClient
}

// NewClient creates a new client wrapping the users service internal client
func NewClient(host string) (*Client, error) {
	usersServiceClient, err := usersclient_internal.NewClient(twitchclient.ClientConf{
		Host: host,
	})

	if err != nil {
		return nil, err
	}

	return &Client{
		usersServiceClient,
	}, nil
}

const timeout = 1 * time.Second

// GetUser returns a User by ID.
func (c *Client) GetUser(ctx context.Context, id string) (types.User, error) {
	var prop *models.Properties

	err := clients.Retry(ctx, clients.NewBackoffWithJitterRetryPolicy(), func() (bool, error) {
		ctxWithTimeout, cancel := context.WithTimeout(ctx, timeout)
		defer cancel()

		var getError error
		prop, getError = c.InternalClient.GetUserByID(ctxWithTimeout, id, nil)

		// Bail out early if the error was a 400 error.
		if getError != nil {
			if e, ok := getError.(client.Error); ok && e.StatusCode() >= 400 && e.StatusCode() < 500 {
				return true, getError
			}
		}

		return false, getError
	})

	if err != nil {
		switch err.(type) {
		case *client.UserNotFoundError:
			return types.User{}, ErrUserNotFound
		default:
			return types.User{}, errors.Wrap(err, "users: failed to get user by id")
		}
	}

	if prop == nil || prop.Login == nil || prop.Displayname == nil {
		return types.User{}, errors.Wrap(err, "users: invalid user object in response")
	}

	// We've observed occasional occurrences of an empty string for the display name from the Users service.
	displayName := *prop.Displayname
	if len(displayName) == 0 {
		displayName = *prop.Login
	}

	return types.User{
		ID:          prop.ID,
		Login:       *prop.Login,
		DisplayName: displayName,
	}, nil
}
