package zuma

import (
	"context"
	"time"

	"code.justin.tv/cb/sauron/types"

	"code.justin.tv/cb/sauron/internal/clients"
	"code.justin.tv/chat/zuma/app/api"
	zumaclient "code.justin.tv/chat/zuma/client"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/foundation/twitchclient"
	"github.com/pkg/errors"
)

// ErrEnforcementDidNotPass occurs when zuma fails enforcement
var ErrEnforcementDidNotPass = errors.New("zuma: message enforcement did not pass")

// Zuma is the interface into zuma functionality
type Zuma interface {
	EnforceMessage(ctx context.Context, channelID, messengerID, message string) (string, []types.Fragment, error)
}

// Client wraps the Zuma service client.
type Client struct {
	zumaClient zumaclient.Client
}

// NewClient creates a new Client.
func NewClient(host string) (*Client, error) {
	zuma, err := zumaclient.NewClient(twitchclient.ClientConf{
		Host: host,
	})

	if err != nil {
		return nil, errors.Wrap(err, "zuma: failed to create new client")
	}

	return &Client{
		zumaClient: zuma,
	}, nil
}

const timeout = 1500 * time.Millisecond

// EnforceMessage calls Zuma's EnforceMessage with retry logic,
// returning only what's needed: the extracted/enforced message and the message fragments
func (c *Client) EnforceMessage(ctx context.Context, channelID, messengerID, message string) (string, []types.Fragment, error) {
	channelEntity := entity.New(entity.NamespaceUser, channelID)

	req := api.EnforceMessageRequest{
		ContainerOwner: &channelEntity,
		EnforcementRules: api.EnforcementRuleset{
			AutoMod:   true,
			Banned:    true,
			Suspended: true,
		},
		MessageText: message,
		SenderID:    messengerID,
	}

	var resp api.EnforceMessageResponse

	err := clients.Retry(ctx, clients.NewBackoffWithJitterRetryPolicy(), func() (bool, error) {
		ctxWithTimeout, cancel := context.WithTimeout(ctx, timeout)
		defer cancel()

		var zumaError error
		resp, zumaError = c.zumaClient.EnforceMessage(ctxWithTimeout, req, nil)
		return false, zumaError
	})

	if err != nil {
		return "", nil, errors.Wrap(err, "zuma: failed to enforce message")
	}

	if !resp.EnforceProperties.AllPassed {
		return "", nil, ErrEnforcementDidNotPass
	}

	var fragments []types.Fragment

	for _, foreignFragment := range resp.ExtractProperties.Content.Fragments {
		if foreignFragment.Emoticon == nil {
			continue
		}

		fragments = append(fragments, types.Fragment{
			Emoticon: types.Emoticon{
				ID:    foreignFragment.Emoticon.EmoticonID,
				SetID: foreignFragment.Emoticon.EmoticonSetID,
			},
			Text: foreignFragment.Text,
		})
	}

	return resp.ExtractProperties.Content.Text, fragments, nil
}
