package convert

import (
	"encoding/base64"
	"time"

	"code.justin.tv/cb/sauron/types"

	"code.justin.tv/cb/sauron/activity/api"
	"code.justin.tv/cb/sauron/internal/clients/dynamodb"
	"github.com/golang/protobuf/ptypes"

	activityType "code.justin.tv/cb/sauron/activity"
	pb "code.justin.tv/cb/sauron/rpc/sauron"
	log "github.com/sirupsen/logrus"
)

// ActivityToPB converts the api activity type to the correct protobuf type
func ActivityToPB(activity api.Activity) *pb.Activity {
	newActivity := &pb.Activity{
		ID:     activity.ID,
		Type:   activity.Type,
		Cursor: activity.Cursor,
	}

	tmsp, err := ptypes.TimestampProto(activity.Timestamp)
	if err != nil {
		log.Warningf("activities: failed to convert timestamp %v to ptypes timestamp", activity.Timestamp)
	} else {
		newActivity.Timestamp = tmsp
	}

	switch activity.Type {
	case activityType.TypeBitsUsage:
		newActivity.Bits = &pb.BitsData{
			BitsAmount:    int64PtrToInt64(activity.BitsAmount),
			BitsAnonymous: boolPtrToBool(activity.BitsAnonymous),
			BitsUserID:    strPtrToStr(activity.BitsUserID),
		}
	case activityType.TypeFollow:
		newActivity.Follows = &pb.FollowData{
			FollowerID: strPtrToStr(activity.FollowerID),
		}
	case activityType.TypeAutoHostStart, activityType.TypeHostStart:
		newActivity.Host = &pb.HostData{
			HostID:             strPtrToStr(activity.HostID),
			HostingViewerCount: intPtrToInt64(activity.HostingViewerCount),
		}
	case activityType.TypeRaiding:
		newActivity.Raid = &pb.RaidData{
			RaiderID: strPtrToStr(activity.RaiderID),
			RaidingViewerCount:  int32PtrToInt32(activity.RaidingViewerCount),
			Status:	int32PtrToInt32(activity.Status),
		}
	case activityType.TypeSubscriptionGiftingIndividual,
		activityType.TypeSubscriptionGiftingCommunity:
		newActivity.SubscriptionGift = &pb.SubscriptionGiftData{
			SubscriptionGifterID:        strPtrToStr(activity.SubscriptionGifterID),
			SubscriptionGiftAnonymous:   boolPtrToBool(activity.SubscriptionGiftAnonymous),
			SubscriptionGiftQuantity:    intPtrToInt64(activity.SubscriptionGiftQuantity),
			SubscriptionGiftRecipientID: strPtrToStr(activity.SubscriptionGiftRecipientID),
			SubscriptionGiftTier:        strPtrToStr(activity.SubscriptionGiftTier),
		}
	case activityType.TypeSubscription,
		activityType.TypeResubscriptionSharing,
		activityType.TypePrimeResubscriptionSharing,
		activityType.TypePrimeSubscription:
		newActivity.Subscription = &pb.SubscriptionData{
			SubscriberID:                       strPtrToStr(activity.SubscriberID),
			SubscriptionTier:                   strPtrToStr(activity.SubscriptionTier),
			SubscriptionCumulativeTenureMonths: intPtrToInt64(activity.SubscriptionCumulativeTenureMonths),
			SubscriptionCustomMessageFragments: fragmentsToPB(activity.SubscriptionCustomMessageFragments),
			SubscriptionCustomMessageText:      strPtrToStr(activity.SubscriptionCustomMessageText),
		}
	}

	return newActivity
}

// DynamoActivityToPB converts a dynamo activity to the correct protobuf type
func DynamoActivityToPB(activity dynamodb.Activity) *pb.Activity {
	newActivity := &pb.Activity{
		ID:     activity.ID,
		Type:   activity.Type,
		Cursor: timeToCursor(activity.Timestamp),
	}

	tmsp, err := ptypes.TimestampProto(activity.Timestamp)
	if err != nil {
		log.Warningf("activities: failed to convert timestamp %v to ptypes timestamp", activity.Timestamp)
	} else {
		newActivity.Timestamp = tmsp
	}

	switch activity.Type {
	case activityType.TypeBitsUsage:
		newActivity.Bits = &pb.BitsData{
			BitsAmount:    int64PtrToInt64(activity.BitsAmount),
			BitsAnonymous: boolPtrToBool(activity.BitsAnonymous),
			BitsUserID:    strPtrToStr(activity.BitsUserID),
		}
	case activityType.TypeFollow:
		newActivity.Follows = &pb.FollowData{
			FollowerID: strPtrToStr(activity.FollowerID),
		}
	case activityType.TypeAutoHostStart, activityType.TypeHostStart:
		newActivity.Host = &pb.HostData{
			HostID:             strPtrToStr(activity.HostID),
			HostingViewerCount: intPtrToInt64(activity.HostingViewerCount),
		}
	case activityType.TypeRaiding:
		newActivity.Raid = &pb.RaidData{
			RaiderID: strPtrToStr(activity.RaiderID),
			RaidingViewerCount:  int32PtrToInt32(activity.RaidingViewerCount),
			Status:	int32PtrToInt32(activity.Status),
		}
	case activityType.TypeSubscriptionGiftingIndividual,
		activityType.TypeSubscriptionGiftingCommunity:
		newActivity.SubscriptionGift = &pb.SubscriptionGiftData{
			SubscriptionGifterID:        strPtrToStr(activity.SubscriptionGifterID),
			SubscriptionGiftAnonymous:   boolPtrToBool(activity.SubscriptionGiftAnonymous),
			SubscriptionGiftQuantity:    intPtrToInt64(activity.SubscriptionGiftQuantity),
			SubscriptionGiftRecipientID: strPtrToStr(activity.SubscriptionGiftRecipientID),
			SubscriptionGiftTier:        strPtrToStr(activity.SubscriptionGiftTier),
		}
	case activityType.TypeSubscription,
		activityType.TypeResubscriptionSharing,
		activityType.TypePrimeResubscriptionSharing,
		activityType.TypePrimeSubscription:
		newActivity.Subscription = &pb.SubscriptionData{
			SubscriberID:                       strPtrToStr(activity.SubscriberID),
			SubscriptionTier:                   strPtrToStr(activity.SubscriptionTier),
			SubscriptionCumulativeTenureMonths: intPtrToInt64(activity.SubscriptionCumulativeTenureMonths),
			SubscriptionCustomMessageFragments: fragmentsToPB(activity.SubscriptionCustomMessageFragments),
			SubscriptionCustomMessageText:      strPtrToStr(activity.SubscriptionCustomMessageText),
		}
	}

	if activity.AlertStatus != nil {
		status := string(*activity.AlertStatus)
		newActivity.AlertStatus = status
	}

	return newActivity
}

// DynamoAlertPrefsToPB converts a dynamo alert prefs type to the correct protobuf type
func DynamoAlertPrefsToPB(alertPrefs *dynamodb.AlertPreferences) *pb.AlertPrefs {
	newAlertPrefs := &pb.AlertPrefs{
		ChannelID:             alertPrefs.ChannelID,
		DNDModeEnabled:        alertPrefs.DNDModeEnabled,
		HideFollows:           alertPrefs.HideFollows,
		HideRaids:             alertPrefs.HideRaids,
		HideHosts:             alertPrefs.HideHosts,
		HideBits:              alertPrefs.HideBits,
		HideSubscriptions:     alertPrefs.HideSubscriptions,
		HideGiftSubscriptions: alertPrefs.HideGiftSubscriptions,
	}

	if alertPrefs.LastModified != nil {
		lastModified, err := ptypes.TimestampProto(*alertPrefs.LastModified)
		if err == nil {
			newAlertPrefs.LastModified = lastModified
		}
	}
	return newAlertPrefs
}

// AlertCursorToTime converts a b64 encoded timestamp string to a time.Time struct
func AlertCursorToTime(cursor string) (time.Time, error) {
	if cursor == "" {
		return time.Now().Add(-(time.Hour * dynamodb.QueueDuration)), nil
	}

	decoded, err := base64.StdEncoding.DecodeString(cursor)
	if err != nil {
		return time.Time{}, err
	}

	parsed, err := time.Parse(time.RFC3339Nano, string(decoded))
	if err != nil {
		return time.Time{}, err
	}

	return parsed, nil
}

func timeToCursor(ts time.Time) string {
	return base64.StdEncoding.EncodeToString([]byte(ts.Format(time.RFC3339Nano)))
}

func intPtrToInt64(cnt *int) int64 {
	if cnt == nil {
		return 0
	}
	return int64(*cnt)
}

func int64PtrToInt64(cnt *int64) int64 {
	if cnt == nil {
		return 0
	}
	return *cnt
}

func int32PtrToInt32(cnt *int32) int32 {
	if cnt == nil {
		return 0
	}
	return *cnt
}

func strPtrToStr(s *string) string {
	if s == nil {
		return ""
	}
	return *s
}

func boolPtrToBool(b *bool) bool {
	if b == nil {
		return false
	}
	return *b
}

func fragmentsToPB(fragments []types.Fragment) []*pb.SubscriptionFragment {
	newFragments := make([]*pb.SubscriptionFragment, len(fragments))
	for idx, fragment := range fragments {
		newFragments[idx] = &pb.SubscriptionFragment{
			Emoticon: &pb.Emoticon{
				ID:    fragment.Emoticon.ID,
				SetID: fragment.Emoticon.SetID,
			},
			Text: fragment.Text,
		}
	}

	return newFragments
}
