package convert_test

import (
	"encoding/base64"
	"testing"
	"time"

	"code.justin.tv/cb/sauron/internal/clients/dynamodb"
	"code.justin.tv/cb/sauron/internal/convert"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
)

func TestAlertCursorToTime(t *testing.T) {
	log.SetLevel(log.PanicLevel)

	Convey("When converting a cursor to a timestamp", t, func() {

		Convey("When given an empty cursor", func() {
			Convey("It returns a timestamp indicating the alert queue duration", func() {
				now := time.Now()

				result, err := convert.AlertCursorToTime("")
				So(err, ShouldBeNil)
				// the result should be a timestamp before the current time
				So(result.Before(now), ShouldBeTrue)
				// the result should be a timestamp after the maximum queue duration
				So(result.After(now.Add(-(time.Hour * dynamodb.QueueDuration))), ShouldBeTrue)
			})
		})

		Convey("When given an invalid b64 encoded string", func() {
			Convey("It should return an error", func() {
				invalidB64 := "asdkj2222"

				result, err := convert.AlertCursorToTime(invalidB64)
				So(result.IsZero(), ShouldBeTrue)
				So(err, ShouldNotBeNil)
			})
		})

		Convey("When given an invalid timestamp", func() {
			Convey("It should return an error", func() {
				invalidTime := base64.StdEncoding.EncodeToString([]byte("2019-01-01,00:00:00,000Z"))

				result, err := convert.AlertCursorToTime(invalidTime)
				So(result.IsZero(), ShouldBeTrue)
				So(err, ShouldNotBeNil)
			})
		})

		Convey("When given an valid cursor", func() {
			Convey("It should return an valid timestamp", func() {
				validTime := base64.StdEncoding.EncodeToString([]byte("2019-01-01T01:01:01.000Z"))

				result, err := convert.AlertCursorToTime(validTime)
				So(result.IsZero(), ShouldBeFalse)
				So(err, ShouldBeNil)
			})
		})
	})
}
