package customrewardredemption_test

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"testing"
	"time"

	"code.justin.tv/cb/sauron/types"

	"code.justin.tv/cb/sauron/internal/event/copo/customrewardredemption"
	"code.justin.tv/cb/sauron/internal/mocks"

	"github.com/aws/aws-lambda-go/events"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

type mockHandlerParams struct {
	DynamoDB *mocks.Database
	Pubsub   *mocks.Publisher
	Users    *mocks.Users
	Statsd   *mocks.StatSender
}

func handlerMocks() mockHandlerParams {
	return mockHandlerParams{
		DynamoDB: &mocks.Database{},
		Pubsub:   &mocks.Publisher{},
		Users:    &mocks.Users{},
		Statsd:   &mocks.StatSender{},
	}
}

func mockHandler(params mockHandlerParams) *customrewardredemption.Handler {
	return &customrewardredemption.Handler{
		DynamoDB: params.DynamoDB,
		Pubsub:   params.Pubsub,
		Users:    params.Users,
		Statsd:   params.Statsd,
	}
}

func getInvokeInput(channelID, redeemerUserID, redemptionID, rewardID, title string, userInput *string) []byte {
	msg := customrewardredemption.Message{
		ChannelID:       channelID,
		RedemptionID:    redemptionID,
		RedeemingUserID: redeemerUserID,
		RewardID:        rewardID,
		RewardTitle:     title,
		UserInput:       userInput,
		Timestamp:       time.Now(),
	}

	msgBytes, _ := json.Marshal(msg)

	snsEntity := events.SNSEntity{
		Message: string(msgBytes),
	}

	snsEntityBytes, _ := json.Marshal(snsEntity)

	sqsMessage := events.SQSMessage{
		Body: string(snsEntityBytes),
	}
	records := make([]events.SQSMessage, 1, 1)
	records[0] = sqsMessage
	input := events.SQSEvent{
		Records: records,
	}

	inputBytes := new(bytes.Buffer)
	_ = json.NewEncoder(inputBytes).Encode(input)
	return inputBytes.Bytes()
}

func TestInvokeWithInvalidInput(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := handlerMocks()
	handler := mockHandler(mocks)
	ctx := context.Background()
	mocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	input := "what is your favorite color"

	Convey("When the handler is invoked with invalid input", t, func() {
		Convey("When the sqs payload contains invalid json", func() {
			Convey("It should return an error", func() {
				invalidSqs := []byte("{\"Records\":}")
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, invalidSqs)
				So(result, ShouldBeNil)
				So(err, ShouldNotBeNil)
			})

		})

		Convey("When the sqs payload has no messages", func() {
			Convey("It should return nil", func() {
				sqs := []byte("{\"Records\":[]}")
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, sqs)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("When the sqs message has invalid json", func() {
			Convey("It should return an error", func() {
				sqs := []byte("{\"Records\":[{\"messageId\":]}")
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, sqs)
				So(result, ShouldBeNil)
				So(err, ShouldNotBeNil)
			})
		})

		Convey("When custom reward redemption event is missing the channel id", func() {
			Convey("It should return a nil result and an error", func() {
				input := getInvokeInput("", "5678", "redemption-123", "reward-123", "Ask me anything", &input)

				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("When custom reward redemption event is missing the redeeming user id", func() {
			Convey("It should return a nil result and an error", func() {
				input := getInvokeInput("1234", "", "redemption-123", "reward-123", "Ask me anything", &input)

				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("When custom reward redemption event is missing the redemption ID", func() {
			Convey("It should return a nil result and an error", func() {
				input := getInvokeInput("1234", "5678", "", "reward-123", "Ask me anything", &input)

				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("When custom reward redemption event is missing the reward title", func() {
			Convey("It should return a nil result and an error", func() {
				input := getInvokeInput("1234", "5678", "redemption-123", "reward-123", "", &input)

				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})
	})
}

func TestInvokeWithValidInput(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := handlerMocks()
	handler := mockHandler(mocks)
	ctx := context.Background()
	redeemingUserID := "111111"
	channelID := "222222"
	redemptionID := "redemption-123"
	rewardID := "reward-123"
	title := "Ask me anything"
	input := "what is your favorite color"

	mocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When the handler is invoked with a valid input", t, func() {
		Convey("When inserting fails", func() {
			Convey("It should return a nil result and an error, and not publish", func() {
				input := getInvokeInput(channelID, redeemingUserID, redemptionID, rewardID, title, &input)
				mocks.Users.On("GetUser", mock.Anything, redeemingUserID).Once().Return(types.User{ID: redeemingUserID, Login: "test", DisplayName: "test"}, nil)

				mocks.DynamoDB.On("InsertChannelPointsCustomRewardRedemption", ctx, channelID, mock.Anything).Once().Return(errors.New("dynamo failed"))
				mocks.Pubsub.AssertNotCalled(t, "PublishChannelPointsCustomRewardRedemption", ctx, channelID, mock.Anything)

				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldNotBeNil)
				mocks.Pubsub.AssertExpectations(t)
			})
		})

		Convey("When inserting succeeds", func() {
			Convey("When publishing to pubsub fails", func() {
				input := getInvokeInput(channelID, redeemingUserID, redemptionID, rewardID, title, &input)
				mocks.Users.On("GetUser", mock.Anything, redeemingUserID).Once().Return(types.User{ID: redeemingUserID, Login: "test", DisplayName: "test"}, nil)

				mocks.DynamoDB.On("InsertChannelPointsCustomRewardRedemption", ctx, channelID, mock.Anything).Once().Return(nil)
				mocks.Pubsub.On("PublishChannelPointsCustomRewardRedemption", ctx, channelID, mock.Anything).Once().Return(errors.New("pubsub failed"))

				Convey("It should return a nil result and an error", func() {
					mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
					result, err := handler.Invoke(ctx, input)
					So(result, ShouldBeNil)
					So(err, ShouldNotBeNil)
				})

				mocks.Pubsub.AssertExpectations(t)
				mocks.DynamoDB.AssertExpectations(t)
			})

			Convey("When publishing to pubsub succeeds", func() {
				input := getInvokeInput(channelID, redeemingUserID, redemptionID, rewardID, title, &input)
				mocks.Users.On("GetUser", mock.Anything, redeemingUserID).Once().Return(types.User{ID: redeemingUserID, Login: "test", DisplayName: "test"}, nil)
				mocks.DynamoDB.On("InsertChannelPointsCustomRewardRedemption", ctx, channelID, mock.Anything).Once().Return(nil)
				mocks.Pubsub.On("PublishChannelPointsCustomRewardRedemption", ctx, channelID, mock.Anything).Once().Return(nil)

				Convey("It should return a nil result and nil error", func() {
					mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
					result, err := handler.Invoke(ctx, input)
					So(result, ShouldBeNil)
					So(err, ShouldBeNil)
				})

				mocks.Pubsub.AssertExpectations(t)
				mocks.DynamoDB.AssertExpectations(t)
			})
		})
	})
}
