package hosting

import (
	"context"

	"code.justin.tv/cb/sauron/internal/clients/users"
	"github.com/aws/aws-lambda-go/events"
	log "github.com/sirupsen/logrus"
)

func (h Handler) processMessage(ctx context.Context, sqsMessage events.SQSMessage) error {
	logger := log.WithField("message_body", sqsMessage.Body)

	msg, err := convert(sqsMessage)
	if err != nil {
		h.Statsd.GoIncrement(errorStat, 1)
		return err
	}

	if err := validate(msg); err != nil {
		logger.WithError(err).Warn("hosting: failed to validate message")
		h.Statsd.GoIncrement(validateErrorStat, 1)
		return nil
	}

	if msg.HostedAfterRaiding {
		logger.Info("hosting: discarding after-raiding event")
		h.Statsd.GoIncrement(statPrefix+"host_after_raiding", 1)
		return nil
	}

	if !msg.ViewerCountRevealed {
		logger.Info("hosting: discarding event with hidden viewer count")
		h.Statsd.GoIncrement(statPrefix+"viewcount_hidden", 1)
		return nil
	}

	if msg.ViewerCount <= 0 {
		logger.Info("hosting: discarding event with zero viewers")
		return nil
	}

	host, err := h.Users.GetUser(ctx, msg.HostChannelID)
	if err != nil {
		switch err {
		case users.ErrUserNotFound:
			h.Statsd.GoIncrement(userNotFoundStat, 1)
			logger.Warnf("hosting: host channel not found for user id '%s'", msg.HostChannelID)
			return nil
		default:
			logger.WithError(err).Errorf("hosting: failed to fetch host hosting channel with user id '%s'", msg.HostChannelID)
			h.Statsd.GoIncrement(errorStat, 1)
			return err
		}
	}

	if msg.AutoHosted {
		return h.insertAndPublishAutoHostStart(ctx, logger, msg.Timestamp, msg.ChannelID, host, &msg.ViewerCount)
	}
	return h.insertAndPublishHostStart(ctx, logger, msg.Timestamp, msg.ChannelID, host, &msg.ViewerCount)
}
