package hypetrain_test

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"testing"
	"time"

	"code.justin.tv/cb/sauron/internal/event/hypetrain"
	"code.justin.tv/cb/sauron/internal/mocks"
	"github.com/aws/aws-lambda-go/events"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

type mockHandlerParams struct {
	DynamoDB *mocks.Database
	Pubsub   *mocks.Publisher
	Users    *mocks.Users
	Statsd   *mocks.StatSender
}

func handlerMocks() mockHandlerParams {
	return mockHandlerParams{
		DynamoDB: &mocks.Database{},
		Pubsub:   &mocks.Publisher{},
		Users:    &mocks.Users{},
		Statsd:   &mocks.StatSender{},
	}
}

func mockHandler(params mockHandlerParams) *hypetrain.Handler {
	return &hypetrain.Handler{
		DynamoDB: params.DynamoDB,
		Pubsub:   params.Pubsub,
		Users:    params.Users,
		Statsd:   params.Statsd,
	}
}

func getInvokeInput(timestamp time.Time, channelID string, hypetrainID string, eventType string) []byte {
	msg := hypetrain.Message{
		Timestamp:   timestamp,
		ChannelID:   channelID,
		HypetrainID: hypetrainID,
		EventType:   eventType,
	}

	msgBytes, _ := json.Marshal(msg)

	snsEntity := events.SNSEntity{
		Message: string(msgBytes),
	}

	snsEntityBytes, _ := json.Marshal(snsEntity)

	sqsMessage := events.SQSMessage{
		Body: string(snsEntityBytes),
	}
	records := make([]events.SQSMessage, 1, 1)
	records[0] = sqsMessage
	input := events.SQSEvent{
		Records: records,
	}

	inputBytes := new(bytes.Buffer)
	_ = json.NewEncoder(inputBytes).Encode(input)
	return inputBytes.Bytes()
}

func TestInvokeWithInvalidInput(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	m := handlerMocks()
	handler := mockHandler(m)
	ctx := context.Background()
	m.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	m.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When the handler is invoked with invalid input", t, func() {
		Convey("When the sqs payload contains invalid json", func() {
			Convey("It should return an error", func() {
				invalidSqs := []byte("{\"Records\":}")
				m.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, invalidSqs)
				So(result, ShouldBeNil)
				So(err, ShouldNotBeNil)
			})

		})

		Convey("When the sqs payload has no messages", func() {
			Convey("It should return nil", func() {
				sqs := []byte("{\"Records\":[]}")
				m.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, sqs)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("When the sqs message has invalid json", func() {
			Convey("It should return an error", func() {
				sqs := []byte("{\"Records\":[{\"messageId\":]}")
				m.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, sqs)
				So(result, ShouldBeNil)
				So(err, ShouldNotBeNil)
			})
		})
	})
}

func TestInvokeWithValidInput(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := handlerMocks()
	handler := mockHandler(mocks)
	ctx := context.Background()
	channelID := "222222"
	hypeTrainID := "abcdef"
	coolDownEventType := "cool_down_expiration"
	timestamp := time.Now()
	mocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("Given HypeTrain handler", t, func() {
		Convey("When inserting fails", func() {
			Convey("It should return a nil result and an error, and not publish", func() {
				input := getInvokeInput(timestamp, channelID, hypeTrainID, coolDownEventType)

				mocks.DynamoDB.On("InsertHypeTrain", ctx, channelID, mock.Anything).Once().Return(errors.New("dynamo failed"))
				mocks.Pubsub.AssertNotCalled(t, "PublishHypeTrain", ctx, channelID, mock.Anything)

				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldNotBeNil)
				mocks.Pubsub.AssertExpectations(t)
			})
		})

		Convey("When inserting succeeds", func() {
			Convey("When publishing to pubsub fails", func() {
				input := getInvokeInput(timestamp, channelID, hypeTrainID, coolDownEventType)

				mocks.DynamoDB.On("InsertHypeTrain", ctx, channelID, mock.Anything).Once().Return(nil)
				mocks.Pubsub.On("PublishHypeTrain", ctx, channelID, mock.Anything).Once().Return(errors.New("pubsub failed"))

				Convey("It should return a nil result and an error", func() {
					mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
					result, err := handler.Invoke(ctx, input)
					So(result, ShouldBeNil)
					So(err, ShouldNotBeNil)
				})

				mocks.Pubsub.AssertExpectations(t)
				mocks.DynamoDB.AssertExpectations(t)
			})

			Convey("When publishing to pubsub succeeds", func() {
				input := getInvokeInput(timestamp, channelID, hypeTrainID, coolDownEventType)

				mocks.DynamoDB.On("InsertHypeTrain", ctx, channelID, mock.Anything).Once().Return(nil)
				mocks.Pubsub.On("PublishHypeTrain", ctx, channelID, mock.Anything).Once().Return(nil)

				Convey("It should return a nil result and nil error", func() {
					mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
					result, err := handler.Invoke(ctx, input)
					So(result, ShouldBeNil)
					So(err, ShouldBeNil)
				})

				mocks.Pubsub.AssertExpectations(t)
				mocks.DynamoDB.AssertExpectations(t)
			})
		})
	})
}
