package fulfillment

import (
	"context"
	"errors"

	"code.justin.tv/cb/sauron/types"

	"code.justin.tv/cb/sauron/internal/clients/users"
	"github.com/aws/aws-lambda-go/events"
	"golang.org/x/sync/errgroup"

	log "github.com/sirupsen/logrus"
)

const (
	// StandardGift is the expected string value for the "FulfillmentType" attribute in the SQS message.
	StandardGift = "StandardGift"
)

var (
	errNonSuccess             = errors.New("fulfillment: received non-successful gift status")
	errInvalidFulfillmentType = errors.New("fulfillment: received non-standard gift (i.e. not individual sub gifting) for fulfillment type")
)

func (h Handler) processMessage(ctx context.Context, sqsMessage events.SQSMessage) error {
	logger := log.WithField("message_body", sqsMessage.Body)

	msg, err := convert(sqsMessage)
	if err != nil {
		switch err {
		case errInvalidFulfillmentType:
			// These errors are expected, so we don't want to error out the lambda and trigger any alarms.
			// Just log it and flag it to statsd
			logger.Info(errInvalidFulfillmentType.Error())
			h.Statsd.GoIncrement(statPrefix+"invalid_type", 1)
			return nil
		default:
			h.Statsd.GoIncrement(errorStat, 1)
			return err
		}
	}

	err = validate(msg)
	if err != nil {
		h.Statsd.GoIncrement(validateErrorStat, 1)
		switch err {
		case errNonSuccess:
			logger.WithError(err).Info("fulfillment: received non-success event")
		default:
			logger.WithError(err).Warn("fulfillment: failed to validate message")
		}
		return nil
	}

	group, groupCtx := errgroup.WithContext(ctx)
	var subTier string
	var recipient, giftSender types.User

	group.Go(func() error {
		var subsError error

		subTier, subsError = h.Subscriptions.GetProductTier(groupCtx, msg.ProductID)
		if subsError != nil {
			logger.WithError(subsError).Errorf("fulfillment: failed to fetch subscription product tier for product id '%s'", msg.ProductID)
			h.Statsd.GoIncrement(errorStat, 1)
			return subsError
		}

		return nil
	})

	group.Go(func() error {
		var recipientError error

		recipient, recipientError = h.Users.GetUser(groupCtx, msg.RecipientID)
		if recipientError != nil {
			switch recipientError {
			case users.ErrUserNotFound:
				logger.Warnf("fulfillment: individual sub gift recipient not found for user id '%s'", msg.RecipientID)
				h.Statsd.GoIncrement(userNotFoundStat, 1)
				return users.ErrUserNotFound
			default:
				logger.WithError(recipientError).Errorf("fulfillment: failed to fetch individual sub gift recipient with user id '%s'", msg.RecipientID)
				h.Statsd.GoIncrement(errorStat, 1)
				return recipientError
			}
		}

		return nil
	})

	group.Go(func() error {
		var senderError error

		giftSender, senderError = h.Users.GetUser(groupCtx, msg.SenderID)
		if senderError != nil {
			switch senderError {
			case users.ErrUserNotFound:
				logger.Warnf("fulfillment: individual sub gifting sender not found for user id '%s'", msg.SenderID)
				h.Statsd.GoIncrement(userNotFoundStat, 1)
				return users.ErrUserNotFound
			default:
				logger.WithError(senderError).Errorf("fulfillment: failed to fetch individual sub gifting sender with user id '%s'", msg.SenderID)
				h.Statsd.GoIncrement(errorStat, 1)
				return senderError
			}
		}

		return nil
	})

	if err := group.Wait(); err != nil {
		switch err {
		case users.ErrUserNotFound:
			return nil
		default:
			return err
		}
	}

	giftedMonths := msg.GiftMonths

	if giftedMonths == 0 {
		giftedMonths = 1
	}

	logger = log.WithFields(log.Fields{
		"fulfillment_msg": msg,
		"gift_sender":     giftSender,
		"recipient":       recipient,
		"sub_tier":        subTier,
		"gift_months":     giftedMonths,
	})

	return h.insertAndPublishSubscriptionGiftingIndividual(ctx, logger, msg.Timestamp, msg.ChannelID, giftSender, msg.AnonGift, recipient, subTier, giftedMonths)
}
