package fulfillment_test

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"testing"
	"time"

	"code.justin.tv/cb/sauron/types"

	"code.justin.tv/cb/sauron/activity"
	"code.justin.tv/cb/sauron/internal/alerts"
	"code.justin.tv/cb/sauron/internal/clients/dynamodb"
	"code.justin.tv/cb/sauron/internal/event/subscription/fulfillment"
	"code.justin.tv/cb/sauron/internal/mocks"

	"github.com/aws/aws-lambda-go/events"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

const targetUserID = "111111"
const fromUserID = "222222"

type mockHandlerParams struct {
	DynamoDB      *mocks.Database
	Pubsub        *mocks.Publisher
	Users         *mocks.Users
	Statsd        *mocks.StatSender
	Subscriptions *mocks.SubService
	AlertManager  *mocks.Manager
}

func handlerMocks() mockHandlerParams {
	return mockHandlerParams{
		DynamoDB:      &mocks.Database{},
		Pubsub:        &mocks.Publisher{},
		Users:         &mocks.Users{},
		Statsd:        &mocks.StatSender{},
		Subscriptions: &mocks.SubService{},
		AlertManager:  &mocks.Manager{},
	}
}

func mockHandler(params mockHandlerParams) *fulfillment.Handler {
	return &fulfillment.Handler{
		DynamoDB:      params.DynamoDB,
		Pubsub:        params.Pubsub,
		Users:         params.Users,
		Statsd:        params.Statsd,
		Subscriptions: params.Subscriptions,
		AlertManager:  params.AlertManager,
	}
}

func getInvokeInput(toID, fromID string, numMonths int) []byte {
	msg := fulfillment.Message{
		AnonGift:    false,
		ChannelID:   toID,
		ProductID:   "1",
		RecipientID: toID,
		SenderID:    fromID,
		Status:      "success",
		Timestamp:   time.Now(),
		GiftMonths:  numMonths,
	}

	msgBytes, _ := json.Marshal(msg)

	snsEntity := events.SNSEntity{
		Message: string(msgBytes),
		MessageAttributes: map[string]interface{}{
			"FulfillmentType": map[string]interface{}{
				"Value": "StandardGift",
			},
		},
	}

	snsEntityBytes, _ := json.Marshal(snsEntity)

	sqsMessage := events.SQSMessage{
		Body: string(snsEntityBytes),
	}
	records := make([]events.SQSMessage, 1, 1)
	records[0] = sqsMessage
	input := events.SQSEvent{
		Records: records,
	}

	inputBytes := new(bytes.Buffer)
	_ = json.NewEncoder(inputBytes).Encode(input)
	return inputBytes.Bytes()
}

func TestInvokeWithInvalidInput(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := handlerMocks()
	handler := mockHandler(mocks)
	ctx := context.Background()
	mocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When the handler is invoked with invalid input", t, func() {
		Convey("When the sqs payload contains invalid json", func() {
			Convey("It should return an error", func() {
				invalidSqs := []byte("{\"Records\":}")
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, invalidSqs)
				So(result, ShouldBeNil)
				So(err, ShouldNotBeNil)
			})
		})

		Convey("When the sqs payload has no messages", func() {
			Convey("It should return nil", func() {
				sqs := []byte("{\"Records\":[]}")
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, sqs)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("When the sqs message has invalid json", func() {
			Convey("It should return an error", func() {
				sqs := []byte("{\"Records\":[{\"messageId\":]}")
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, sqs)
				So(result, ShouldBeNil)
				So(err, ShouldNotBeNil)
			})
		})

		Convey("When fulfillment event is missing the recipient id", func() {
			Convey("It should return a nil result and an error", func() {
				input := getInvokeInput("", fromUserID, 1)

				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("When fulfillment event is missing the gifter id", func() {
			Convey("It should return a nil result and an error", func() {
				input := getInvokeInput(targetUserID, "", 1)

				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})
	})
}

func TestInvokeWithValidInput(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := handlerMocks()
	handler := mockHandler(mocks)
	ctx := context.Background()
	alertStatus := alerts.Status{
		CanPublish: true,
		StatusName: dynamodb.AlertStatusQueued,
	}

	productID := "1"
	mocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When the handler is invoked with a valid input", t, func() {
		Convey("When inserting fails", func() {
			Convey("It should return a nil result and an error, and not publish", func() {
				input := getInvokeInput(targetUserID, fromUserID, 1)
				mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
				mocks.Users.On("GetUser", mock.Anything, targetUserID).Once().Return(types.User{ID: targetUserID, Login: "test", DisplayName: "test"}, nil)
				mocks.Subscriptions.On("GetProductTier", mock.Anything, productID).Once().Return("sub-tier", nil)

				mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeSubscriptionGiftingIndividual).Once().Return(alertStatus, nil)

				mocks.DynamoDB.On("GetAlertPreferences", ctx, targetUserID).Once().Return(&dynamodb.AlertPreferences{}, nil)
				mocks.DynamoDB.On("InsertSubscriptionGiftingIndividual", ctx, mock.Anything, mock.Anything).Once().Return(errors.New("dynamo failed"))

				mocks.Pubsub.AssertNotCalled(t, "PublishSubscriptionGiftingIndividual", ctx, mock.Anything, mock.Anything, mock.Anything)

				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldNotBeNil)
				mocks.Pubsub.AssertExpectations(t)
			})
		})

		Convey("When inserting succeeds", func() {
			Convey("When publishing to pubsub fails", func() {
				input := getInvokeInput(targetUserID, fromUserID, 1)
				mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
				mocks.Users.On("GetUser", mock.Anything, targetUserID).Once().Return(types.User{ID: targetUserID, Login: "test", DisplayName: "test"}, nil)
				mocks.Subscriptions.On("GetProductTier", mock.Anything, productID).Once().Return("sub-tier", nil)

				mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeSubscriptionGiftingIndividual).Once().Return(alertStatus, nil)

				mocks.DynamoDB.On("GetAlertPreferences", ctx, targetUserID).Once().Return(&dynamodb.AlertPreferences{}, nil)
				mocks.DynamoDB.On("InsertSubscriptionGiftingIndividual", ctx, mock.Anything, mock.Anything).Once().Return(nil)

				mocks.Pubsub.On("PublishSubscriptionGiftingIndividual", ctx, mock.Anything, mock.Anything, mock.Anything).Once().Return(errors.New("pubsub failed"))

				Convey("It should return a nil result and an error", func() {
					mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
					result, err := handler.Invoke(ctx, input)
					So(result, ShouldBeNil)
					So(err, ShouldNotBeNil)
				})
			})

			Convey("When publishing to pubsub succeeds", func() {
				input := getInvokeInput(targetUserID, fromUserID, 1)
				mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
				mocks.Users.On("GetUser", mock.Anything, targetUserID).Once().Return(types.User{ID: targetUserID, Login: "test", DisplayName: "test"}, nil)
				mocks.Subscriptions.On("GetProductTier", mock.Anything, productID).Once().Return("sub-tier", nil)

				mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeSubscriptionGiftingIndividual).Once().Return(alertStatus, nil)

				mocks.DynamoDB.On("InsertSubscriptionGiftingIndividual", ctx, mock.Anything, mock.Anything).Once().Return(nil)

				mocks.Pubsub.On("PublishSubscriptionGiftingIndividual", ctx, mock.Anything, mock.Anything, mock.Anything).Once().Return(nil)
				mocks.Pubsub.On("PublishAlert", ctx, targetUserID, mock.Anything).Once().Return(nil)

				Convey("It should return a nil result and nil error", func() {
					mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
					result, err := handler.Invoke(ctx, input)
					So(result, ShouldBeNil)
					So(err, ShouldBeNil)
				})
			})
		})
	})
}

func TestInvokeWithAlertPublishing(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := handlerMocks()
	handler := mockHandler(mocks)
	ctx := context.Background()
	alertStatus := alerts.Status{
		CanPublish: false,
		StatusName: dynamodb.AlertStatusQueued,
	}

	productID := "1"
	mocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When the handler is successfully invoked", t, func() {
		Convey("It should not publish when the alerts manager says not to", func() {
			input := getInvokeInput(targetUserID, fromUserID, 1)
			mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
			mocks.Users.On("GetUser", mock.Anything, targetUserID).Once().Return(types.User{ID: targetUserID, Login: "test", DisplayName: "test"}, nil)
			mocks.Subscriptions.On("GetProductTier", mock.Anything, productID).Once().Return("sub-tier", nil)

			mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeSubscriptionGiftingIndividual).Once().Return(alertStatus, nil)

			mocks.DynamoDB.On("InsertSubscriptionGiftingIndividual", ctx, mock.Anything, mock.Anything).Once().Return(nil)

			mocks.Pubsub.On("PublishSubscriptionGiftingIndividual", ctx, mock.Anything, mock.Anything, mock.Anything).Once().Return(nil)

			mocks.Pubsub.AssertNotCalled(t, "PublishAlert", ctx, targetUserID, mock.Anything)

			Convey("It should not publish an alert", func() {
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
				mocks.AlertManager.AssertExpectations(t)
			})
		})

		Convey("It should publish when alert manager says to", func() {
			input := getInvokeInput(targetUserID, fromUserID, 1)
			alertStatusSuccess := alerts.Status{
				CanPublish: true,
				StatusName: dynamodb.AlertStatusQueued,
			}
			mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
			mocks.Users.On("GetUser", mock.Anything, targetUserID).Once().Return(types.User{ID: targetUserID, Login: "test", DisplayName: "test"}, nil)
			mocks.Subscriptions.On("GetProductTier", mock.Anything, productID).Once().Return("sub-tier", nil)

			mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeSubscriptionGiftingIndividual).Once().Return(alertStatusSuccess, nil)

			mocks.DynamoDB.On("InsertSubscriptionGiftingIndividual", ctx, mock.Anything, mock.Anything).Once().Return(nil)

			mocks.Pubsub.On("PublishSubscriptionGiftingIndividual", ctx, mock.Anything, mock.Anything, mock.Anything).Once().Return(nil)
			mocks.Pubsub.On("PublishAlert", ctx, targetUserID, mock.Anything).Once().Return(nil)

			Convey("It should not publish an alert", func() {
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})
	})
}

func TestGiftedMonthsDefaultWorksAsExpected(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := handlerMocks()
	handler := mockHandler(mocks)
	ctx := context.Background()
	productID := "1"
	mocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When the handler is successfully invoked", t, func() {
		Convey("With 0 months set, it should default to 1", func() {
			input := getInvokeInput(targetUserID, fromUserID, 0)
			alertStatusSuccess := alerts.Status{
				CanPublish: true,
				StatusName: dynamodb.AlertStatusQueued,
			}
			mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
			mocks.Users.On("GetUser", mock.Anything, targetUserID).Once().Return(types.User{ID: targetUserID, Login: "test", DisplayName: "test"}, nil)
			mocks.Subscriptions.On("GetProductTier", mock.Anything, productID).Once().Return("sub-tier", nil)

			mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeSubscriptionGiftingIndividual).Once().Return(alertStatusSuccess, nil)

			mocks.DynamoDB.On("InsertSubscriptionGiftingIndividual", ctx, mock.Anything, mock.Anything).Once().Return(nil)

			mocks.Pubsub.On("PublishSubscriptionGiftingIndividual", ctx, mock.Anything, mock.Anything, mock.Anything).Once().Return(nil)
			mocks.Pubsub.On("PublishAlert", ctx, targetUserID, mock.Anything).Once().Return(nil)

			Convey("It should not publish an alert", func() {
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)

				actualPubSubInput := mocks.DynamoDB.Calls[0].Arguments.Get(2).(dynamodb.SubscriptionGiftingIndividual)
				So(actualPubSubInput.MonthAmount, ShouldEqual, int32(1))
			})
		})
		Convey("With 1 months set, it should still be set to 1", func() {
			input := getInvokeInput(targetUserID, fromUserID, 1)
			alertStatusSuccess := alerts.Status{
				CanPublish: true,
				StatusName: dynamodb.AlertStatusQueued,
			}
			mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
			mocks.Users.On("GetUser", mock.Anything, targetUserID).Once().Return(types.User{ID: targetUserID, Login: "test", DisplayName: "test"}, nil)
			mocks.Subscriptions.On("GetProductTier", mock.Anything, productID).Once().Return("sub-tier", nil)

			mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeSubscriptionGiftingIndividual).Once().Return(alertStatusSuccess, nil)

			mocks.DynamoDB.On("InsertSubscriptionGiftingIndividual", ctx, mock.Anything, mock.Anything).Once().Return(nil)

			mocks.Pubsub.On("PublishSubscriptionGiftingIndividual", ctx, mock.Anything, mock.Anything, mock.Anything).Once().Return(nil)
			mocks.Pubsub.On("PublishAlert", ctx, targetUserID, mock.Anything).Once().Return(nil)

			Convey("It should not publish an alert", func() {
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)

				actualPubSubInput := mocks.DynamoDB.Calls[1].Arguments.Get(2).(dynamodb.SubscriptionGiftingIndividual)
				So(actualPubSubInput.MonthAmount, ShouldEqual, int32(1))
			})
		})
		Convey("With 12 months set, it should still be set to 12", func() {
			input := getInvokeInput(targetUserID, fromUserID, 12)
			alertStatusSuccess := alerts.Status{
				CanPublish: true,
				StatusName: dynamodb.AlertStatusQueued,
			}
			mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
			mocks.Users.On("GetUser", mock.Anything, targetUserID).Once().Return(types.User{ID: targetUserID, Login: "test", DisplayName: "test"}, nil)
			mocks.Subscriptions.On("GetProductTier", mock.Anything, productID).Once().Return("sub-tier", nil)

			mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeSubscriptionGiftingIndividual).Once().Return(alertStatusSuccess, nil)

			mocks.DynamoDB.On("InsertSubscriptionGiftingIndividual", ctx, mock.Anything, mock.Anything).Once().Return(nil)

			mocks.Pubsub.On("PublishSubscriptionGiftingIndividual", ctx, mock.Anything, mock.Anything, mock.Anything).Once().Return(nil)
			mocks.Pubsub.On("PublishAlert", ctx, targetUserID, mock.Anything).Once().Return(nil)

			Convey("It should not publish an alert", func() {
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)

				actualPubSubInput := mocks.DynamoDB.Calls[2].Arguments.Get(2).(dynamodb.SubscriptionGiftingIndividual)
				So(actualPubSubInput.MonthAmount, ShouldEqual, int32(12))
			})
		})

	})
}
