package mmg

import (
	"context"
	"errors"

	"code.justin.tv/cb/sauron/internal/clients/users"
	"code.justin.tv/cb/sauron/types"
	"github.com/aws/aws-lambda-go/events"
	"golang.org/x/sync/errgroup"

	log "github.com/sirupsen/logrus"
)

var errNonGiftStatus = errors.New("mmg: received non-start gift status")

func (h Handler) processMessage(ctx context.Context, sqsMessage events.SQSMessage) error {
	logger := log.WithField("message_body", sqsMessage.Body)

	msg, err := convert(sqsMessage)
	if err != nil {
		h.Statsd.GoIncrement(errorStat, 1)
		return err
	}

	err = validate(msg)
	if err != nil {
		h.Statsd.GoIncrement(validateErrorStat, 1)
		switch err {
		case errNonGiftStatus:
			logger.WithError(err).Info("mmg: found non gift status in message")
		default:
			logger.WithError(err).Warn("mmg: failed to validate mmg message")
		}
		return nil
	}

	group, groupCtx := errgroup.WithContext(ctx)
	var subTier string
	var giftSender types.User

	group.Go(func() error {
		var subsError error

		subTier, subsError = h.Subscriptions.GetProductTier(groupCtx, msg.ProductID)
		if subsError != nil {
			logger.WithError(subsError).Errorf("mmg: failed to fetch subscription product tier for product id '%s'", msg.ProductID)
			h.Statsd.GoIncrement(errorStat, 1)
			return subsError
		}

		return nil
	})

	group.Go(func() error {
		var senderError error

		giftSender, senderError = h.Users.GetUser(groupCtx, msg.SenderID)
		if senderError != nil {
			switch senderError {
			case users.ErrUserNotFound:
				logger.Warnf("mmg: community sub gifting sender not found for user id '%s'", msg.SenderID)
				h.Statsd.GoIncrement(userNotFoundStat, 1)
				return users.ErrUserNotFound
			default:
				logger.WithError(senderError).Errorf("mmg: failed to fetch community sub gifting sender with user id '%s'", msg.SenderID)
				h.Statsd.GoIncrement(errorStat, 1)
				return senderError
			}
		}

		return nil
	})

	if err := group.Wait(); err != nil {
		switch err {
		case users.ErrUserNotFound:
			return nil
		default:
			return err
		}
	}

	giftedMonths := msg.GiftMonths

	if giftedMonths == 0 {
		giftedMonths = 1
	}

	logger = log.WithFields(log.Fields{
		"mmg_msg":     msg,
		"gift_sender": giftSender,
		"sub_tier":    subTier,
		"gift_months": giftedMonths,
	})

	return h.insertAndPublishSubscriptionGiftingCommunity(ctx, logger, msg.Timestamp, msg.ChannelID, giftSender, msg.AnonGift, msg.GiftQuantity, subTier, giftedMonths)
}
