package mmg_test

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"testing"
	"time"

	"code.justin.tv/cb/sauron/types"

	"code.justin.tv/cb/sauron/activity"
	"code.justin.tv/cb/sauron/internal/alerts"
	"code.justin.tv/cb/sauron/internal/clients/dynamodb"
	"code.justin.tv/cb/sauron/internal/event/subscription/mmg"
	"code.justin.tv/cb/sauron/internal/mocks"

	"github.com/aws/aws-lambda-go/events"
	log "github.com/sirupsen/logrus"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

type mockHandlerParams struct {
	DynamoDB      *mocks.Database
	Pubsub        *mocks.Publisher
	Users         *mocks.Users
	Statsd        *mocks.StatSender
	Subscriptions *mocks.SubService
	AlertManager  *mocks.Manager
}

func handlerMocks() mockHandlerParams {
	return mockHandlerParams{
		DynamoDB:      &mocks.Database{},
		Pubsub:        &mocks.Publisher{},
		Users:         &mocks.Users{},
		Statsd:        &mocks.StatSender{},
		Subscriptions: &mocks.SubService{},
		AlertManager:  &mocks.Manager{},
	}
}

func mockHandler(params mockHandlerParams) *mmg.Handler {
	return &mmg.Handler{
		DynamoDB:      params.DynamoDB,
		Pubsub:        params.Pubsub,
		Users:         params.Users,
		Statsd:        params.Statsd,
		Subscriptions: params.Subscriptions,
		AlertManager:  params.AlertManager,
	}
}

func getInvokeInput(toID, fromID, status, productID string, quantity int) []byte {
	msg := mmg.Message{
		AnonGift:     false,
		ChannelID:    toID,
		GiftQuantity: quantity,
		ProductID:    productID,
		SenderID:     fromID,
		Status:       status,
		Timestamp:    time.Now(),
	}

	msgBytes, _ := json.Marshal(msg)

	snsEntity := events.SNSEntity{
		Message: string(msgBytes),
	}

	snsEntityBytes, _ := json.Marshal(snsEntity)

	sqsMessage := events.SQSMessage{
		Body: string(snsEntityBytes),
	}
	records := make([]events.SQSMessage, 1, 1)
	records[0] = sqsMessage
	input := events.SQSEvent{
		Records: records,
	}

	inputBytes := new(bytes.Buffer)
	_ = json.NewEncoder(inputBytes).Encode(input)
	return inputBytes.Bytes()
}

func TestInvokeWithInvalidInput(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := handlerMocks()
	handler := mockHandler(mocks)
	ctx := context.Background()
	mocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When the handler is invoked with invalid input", t, func() {
		Convey("When the sqs payload contains invalid json", func() {
			Convey("It should return an error", func() {
				invalidSqs := []byte("{\"Records\":}")
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, invalidSqs)
				So(result, ShouldBeNil)
				So(err, ShouldNotBeNil)
			})

		})

		Convey("When the sqs payload has no messages", func() {
			Convey("It should return nil", func() {
				sqs := []byte("{\"Records\":[]}")
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, sqs)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("When the sqs message has invalid json", func() {
			Convey("It should return an error", func() {
				sqs := []byte("{\"Records\":[{\"messageId\":]}")
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, sqs)
				So(result, ShouldBeNil)
				So(err, ShouldNotBeNil)
			})
		})

		Convey("When mmg event is missing the channel id", func() {
			Convey("It should return a nil result and an error", func() {
				input := getInvokeInput("", "222222", "start", "1", 10)

				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("When mmg event is missing the sender id", func() {
			Convey("It should return a nil result and an error", func() {
				input := getInvokeInput("111111", "", "start", "1", 10)

				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("When mmg event is missing the product id", func() {
			Convey("It should return a nil result and an error", func() {
				input := getInvokeInput("111111", "", "start", "", 10)

				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("When mmg event has 0 gift quantity", func() {
			Convey("It should return a nil result and an error", func() {
				input := getInvokeInput("111111", "", "start", "1", 0)

				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("When mmg event has invalid start", func() {
			Convey("It should return a nil result and an nil error", func() {
				input := getInvokeInput("111111", "", "not-start", "1", 0)

				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})
	})
}

func TestInvokeWithValidInput(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := handlerMocks()
	handler := mockHandler(mocks)
	ctx := context.Background()
	alertStatus := alerts.Status{
		CanPublish: true,
		StatusName: dynamodb.AlertStatusQueued,
	}
	targetUserID := "111111"
	fromUserID := "222222"
	status := "start"
	productID := "1"
	quantity := 10

	mocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When the handler is invoked with a valid input", t, func() {
		Convey("When inserting fails", func() {
			Convey("It should return a nil result and an error, and not publish", func() {
				input := getInvokeInput(targetUserID, fromUserID, status, productID, quantity)
				mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
				mocks.Users.On("GetUser", mock.Anything, targetUserID).Once().Return(types.User{ID: targetUserID, Login: "test", DisplayName: "test"}, nil)
				mocks.Subscriptions.On("GetProductTier", mock.Anything, productID).Once().Return("sub-tier", nil)

				mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeSubscriptionGiftingCommunity).Once().Return(alertStatus, nil)

				mocks.DynamoDB.On("GetAlertPreferences", ctx, targetUserID).Once().Return(&dynamodb.AlertPreferences{}, nil)
				mocks.DynamoDB.On("InsertSubscriptionGiftingCommunity", ctx, targetUserID, mock.Anything).Once().Return(errors.New("dynamo failed"))

				mocks.Pubsub.AssertNotCalled(t, "PublishSubscriptionGiftingCommunity", ctx, targetUserID, mock.Anything, mock.Anything)

				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldNotBeNil)
				mocks.Pubsub.AssertExpectations(t)
			})
		})

		Convey("When inserting succeeds", func() {
			Convey("When publishing to pubsub fails", func() {
				input := getInvokeInput(targetUserID, fromUserID, status, productID, quantity)
				mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
				mocks.Users.On("GetUser", mock.Anything, targetUserID).Once().Return(types.User{ID: targetUserID, Login: "test", DisplayName: "test"}, nil)
				mocks.Subscriptions.On("GetProductTier", mock.Anything, productID).Once().Return("sub-tier", nil)

				mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeSubscriptionGiftingCommunity).Once().Return(alertStatus, nil)

				mocks.DynamoDB.On("GetAlertPreferences", ctx, targetUserID).Once().Return(&dynamodb.AlertPreferences{}, nil)
				mocks.DynamoDB.On("InsertSubscriptionGiftingCommunity", ctx, targetUserID, mock.Anything).Once().Return(nil)

				mocks.Pubsub.On("PublishSubscriptionGiftingCommunity", ctx, targetUserID, mock.Anything, mock.Anything).Once().Return(errors.New("pubsub failed"))

				Convey("It should return a nil result and an error", func() {
					mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
					result, err := handler.Invoke(ctx, input)
					So(result, ShouldBeNil)
					So(err, ShouldNotBeNil)
				})
			})

			Convey("When publishing to pubsub succeeds", func() {
				input := getInvokeInput(targetUserID, fromUserID, status, productID, quantity)
				mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
				mocks.Users.On("GetUser", mock.Anything, targetUserID).Once().Return(types.User{ID: targetUserID, Login: "test", DisplayName: "test"}, nil)
				mocks.Subscriptions.On("GetProductTier", mock.Anything, productID).Once().Return("sub-tier", nil)

				mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeSubscriptionGiftingCommunity).Once().Return(alertStatus, nil)

				mocks.DynamoDB.On("InsertSubscriptionGiftingCommunity", ctx, targetUserID, mock.Anything).Once().Return(nil)

				mocks.Pubsub.On("PublishSubscriptionGiftingCommunity", ctx, targetUserID, mock.Anything, mock.Anything).Once().Return(nil)
				mocks.Pubsub.On("PublishAlert", ctx, targetUserID, mock.Anything).Once().Return(nil)

				Convey("It should return a nil result and nil error", func() {
					mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
					result, err := handler.Invoke(ctx, input)
					So(result, ShouldBeNil)
					So(err, ShouldBeNil)
				})
			})
		})
	})
}

func TestInvokeWithAlertPublishing(t *testing.T) {
	log.SetLevel(log.PanicLevel)
	mocks := handlerMocks()
	handler := mockHandler(mocks)
	ctx := context.Background()
	alertStatus := alerts.Status{
		CanPublish: false,
		StatusName: dynamodb.AlertStatusQueued,
	}
	targetUserID := "111111"
	fromUserID := "222222"
	status := "start"
	productID := "1"
	quantity := 10

	mocks.Statsd.On("GoIncrement", mock.Anything, mock.Anything).Maybe()
	mocks.Statsd.On("GoExecutionTime", mock.Anything, mock.Anything).Maybe()

	Convey("When the handler is successfully invoked", t, func() {
		Convey("It should not publish when the alerts manager says not to", func() {
			input := getInvokeInput(targetUserID, fromUserID, status, productID, quantity)
			mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
			mocks.Users.On("GetUser", mock.Anything, targetUserID).Once().Return(types.User{ID: targetUserID, Login: "test", DisplayName: "test"}, nil)
			mocks.Subscriptions.On("GetProductTier", mock.Anything, productID).Once().Return("sub-tier", nil)

			mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeSubscriptionGiftingCommunity).Once().Return(alertStatus, nil)

			mocks.DynamoDB.On("InsertSubscriptionGiftingCommunity", ctx, mock.Anything, mock.Anything).Once().Return(nil)

			mocks.Pubsub.AssertNotCalled(t, "PublishAlert", ctx, targetUserID, mock.Anything)
			mocks.Pubsub.On("PublishSubscriptionGiftingCommunity", ctx, mock.Anything, mock.Anything, mock.Anything).Once().Return(nil)

			Convey("It should not publish an alert", func() {
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
				mocks.AlertManager.AssertExpectations(t)
			})
		})

		Convey("It should publish when alert manager says to", func() {
			input := getInvokeInput(targetUserID, fromUserID, status, productID, quantity)
			alertStatusSuccess := alerts.Status{
				CanPublish: true,
				StatusName: dynamodb.AlertStatusQueued,
			}
			mocks.Users.On("GetUser", mock.Anything, fromUserID).Once().Return(types.User{ID: fromUserID, Login: "test", DisplayName: "test"}, nil)
			mocks.Users.On("GetUser", mock.Anything, targetUserID).Once().Return(types.User{ID: targetUserID, Login: "test", DisplayName: "test"}, nil)
			mocks.Subscriptions.On("GetProductTier", mock.Anything, productID).Once().Return("sub-tier", nil)

			mocks.AlertManager.On("GetAlertStatus", ctx, targetUserID, activity.TypeSubscriptionGiftingCommunity).Once().Return(alertStatusSuccess, nil)

			mocks.DynamoDB.On("InsertSubscriptionGiftingCommunity", ctx, mock.Anything, mock.Anything).Once().Return(nil)

			mocks.Pubsub.On("PublishSubscriptionGiftingCommunity", ctx, mock.Anything, mock.Anything, mock.Anything).Once().Return(nil)
			mocks.Pubsub.On("PublishAlert", ctx, targetUserID, mock.Anything).Once().Return(nil)

			Convey("It should not publish an alert", func() {
				mocks.Statsd.On("Shutdown", mock.Anything).Once().Return(nil)
				result, err := handler.Invoke(ctx, input)
				So(result, ShouldBeNil)
				So(err, ShouldBeNil)
			})
		})
	})
}
