package notice

import (
	"context"

	"code.justin.tv/cb/sauron/types"

	"code.justin.tv/cb/sauron/internal/clients/users"
	"code.justin.tv/cb/sauron/internal/clients/zuma"
	"github.com/aws/aws-lambda-go/events"
	"golang.org/x/sync/errgroup"

	log "github.com/sirupsen/logrus"
)

// SubPlanPrime is the subscription product plan for Twitch Prime.
const SubPlanPrime = "Prime"

func (h Handler) processMessage(ctx context.Context, sqsMessage events.SQSMessage) error {
	logger := log.WithField("message_body", sqsMessage.Body)

	msg, err := convert(sqsMessage)
	if err != nil {
		h.Statsd.GoIncrement(errorStat, 1)
		return err
	}

	err = validate(msg)
	if err != nil {
		h.Statsd.GoIncrement(validateErrorStat, 1)
		logger.WithError(err).Warn("notice: failed to validate message")
		return nil
	}

	if *msg.CumulativeTenureMonths > 1 {
		return h.processAnyResubscriptionSharing(ctx, msg, logger)
	}

	return h.processAnySubscription(ctx, msg, logger)
}

func (h Handler) processAnyResubscriptionSharing(ctx context.Context, resub Message, logger *log.Entry) error {
	group, groupCtx := errgroup.WithContext(ctx)
	var subscriber types.User

	group.Go(func() error {
		var usersError error

		subscriber, usersError = h.Users.GetUser(groupCtx, resub.UserID)
		if usersError != nil {
			switch usersError {
			case users.ErrUserNotFound:
				logger.Warnf("notice: subscriber not found for user id '%s'", resub.UserID)
				return users.ErrUserNotFound
			default:
				logger.WithError(usersError).Errorf("notice: failed to fetch subscriber with user id '%s'", resub.UserID)
				return usersError
			}
		}

		return nil
	})

	var enforcedMessage string
	var fragments []types.Fragment

	if len(resub.CustomMessage) > 0 {
		group.Go(func() error {
			var zumaError error

			enforcedMessage, fragments, zumaError = h.Zuma.EnforceMessage(groupCtx, resub.ChannelID, resub.UserID, resub.CustomMessage)
			if zumaError != nil {
				switch zumaError {
				case zuma.ErrEnforcementDidNotPass:
					logger.Info("notice: zuma's message enforcement did not pass")
					h.Statsd.GoIncrement(errorStat, 1)
					return zuma.ErrEnforcementDidNotPass
				default:
					logger.WithError(zumaError).Error("notice: failed to enforce message via zuma")
					h.Statsd.GoIncrement(errorStat, 1)
					return zumaError
				}
			}

			return nil
		})
	}

	if err := group.Wait(); err != nil {
		switch err {
		case users.ErrUserNotFound:
			h.Statsd.GoIncrement(userNotFoundStat, 1)
			return nil
		case zuma.ErrEnforcementDidNotPass:
			// Mimic the behavior of the Subscriptions service:
			// https://git.xarth.tv/revenue/subscriptions/blob/c9f79cfb54506a2da9b61f766ed8d85b7e28aafc/internal/app/chatnotification/chatnotification.go#L300-L303
			h.Statsd.GoIncrement(errorStat, 1)
			return nil
		default:
			return err
		}
	}

	logger = log.WithFields(log.Fields{
		"resub":     resub,
		"fragments": fragments,
	})
	if resub.SubPlan == SubPlanPrime {
		return h.insertAndPublishPrimeResubscriptionSharing(ctx, logger, resub.Timestamp, resub.ChannelID, subscriber, *resub.CumulativeTenureMonths, fragments, enforcedMessage, resub.MultiMonthDuration, resub.MultiMonthTenure)
	}
	return h.insertAndPublishResubscriptionSharing(ctx, logger, resub.Timestamp, resub.ChannelID, subscriber, resub.Tier, *resub.CumulativeTenureMonths, fragments, enforcedMessage, resub.MultiMonthDuration, resub.MultiMonthTenure)
}

func (h Handler) processAnySubscription(ctx context.Context, sub Message, logger *log.Entry) error {
	subscriber, err := h.Users.GetUser(ctx, sub.UserID)
	if err != nil {
		switch err {
		case users.ErrUserNotFound:
			h.Statsd.GoIncrement(errorStat, 1)
			logger.Warnf("notice: subscriber not found for user id '%s'", sub.UserID)
			return nil
		default:
			h.Statsd.GoIncrement(errorStat, 1)
			logger.WithError(err).Errorf("notice: failed to fetch subscriber with user id '%s'", sub.UserID)
			return err
		}
	}

	logger = log.WithFields(log.Fields{
		"sub_event":  sub,
		"subscriber": subscriber,
	})

	if sub.SubPlan == SubPlanPrime {
		return h.insertAndPublishPrimeSubscription(ctx, logger, sub.Timestamp, sub.ChannelID, subscriber)
	}
	return h.insertAndPublishSubscription(ctx, logger, sub.Timestamp, sub.ChannelID, subscriber, sub.Tier, sub.MultiMonthDuration, sub.MultiMonthTenure)
}
